/************************************************************************/
/*                                                                      */
/*    *****                       *****                                 */
/*      *****                   *****                                   */
/*        *****               *****                                     */
/*          *****           *****                                       */
/*  ***************       ***************                               */
/*  *****************   *****************                               */
/*  ***************       ***************                               */
/*          *****           *****           TheNetNode                  */
/*        *****               *****         Portable                    */
/*      *****                   *****       Network                     */
/*    *****                       *****     Software                    */
/*                                                                      */
/* File os/linux/kernelif.c (maintained by: DG9OBU)                     */
/*                                                                      */
/* This file is part of "TheNetNode" - Software Package                 */
/*                                                                      */
/* Copyright (C) 1998 - 2003 NORD><LINK e.V. Braunschweig               */
/*                                                                      */
/* This program is free software; you can redistribute it and/or modify */
/* it under the terms of the NORD><LINK ALAS (Allgemeine Lizenz fuer    */
/* Amateurfunk Software) as published by Hans Georg Giese (DF2AU)       */
/* on 13/Oct/1992; either version 1, or (at your option) any later      */
/* version.                                                             */
/*                                                                      */
/* This program is distributed WITHOUT ANY WARRANTY only for further    */
/* development and learning purposes. See the ALAS (Allgemeine Lizenz   */
/* fuer Amateurfunk Software).                                          */
/*                                                                      */
/* You should have received a copy of the NORD><LINK ALAS (Allgemeine   */
/* Lizenz fuer Amateurfunk Software) along with this program; if not,   */
/* write to NORD><LINK e.V., Hinter dem Berge 5, D-38108 Braunschweig   */
/*                                                                      */
/* Dieses Programm ist PUBLIC DOMAIN, mit den Einschraenkungen durch    */
/* die ALAS (Allgemeine Lizenz fuer Amateurfunk Software), entweder     */
/* Version 1, veroeffentlicht von Hans Georg Giese (DF2AU),             */
/* am 13.Oct.1992, oder (wenn gewuenscht) jede spaetere Version.        */
/*                                                                      */
/* Dieses Programm wird unter Haftungsausschluss vertrieben, aus-       */
/* schliesslich fuer Weiterentwicklungs- und Lehrzwecke. Naeheres       */
/* koennen Sie der ALAS (Allgemeine Lizenz fuer Amateurfunk Software)   */
/* entnehmen.                                                           */
/*                                                                      */
/* Sollte dieser Software keine ALAS (Allgemeine Lizenz fuer Amateur-   */
/* funk Software) beigelegen haben, wenden Sie sich bitte an            */
/* NORD><LINK e.V., Hinter dem Berge 5, D-38108 Braunschweig            */
/*                                                                      */
/************************************************************************/

#include "tnn.h"

#ifdef KERNELIF
#include "kernelif.h"

struct kip kip_info;

/* Funktionen fuer beide Teile */
static void show_error(MBHEAD *, int);

/* Funktionen fuer den IP-Teil */

static void ifip_close(struct kip *);
static BOOLEAN ifip_setup(struct kip *);
static void ifip_init(struct kip *);
static BOOLEAN ifip_usable(struct kip *);
static BOOLEAN ifip_kernelip(struct kip *, char *);

/* Funktionen fuer den AX.25-Teil */

BOOLEAN ifax_setup(DEVICE *);
void ifax_close(DEVICE *);
void ifax_rx(int);
void ifax_tx(void);
void ifax_housekeeping(void);

BOOLEAN update_parms = FALSE;

/**************************************************************************/
/* Funktion zur Anzeige der Fehlernummer und des Fehlertextes             */
/*------------------------------------------------------------------------*/
void show_error(MBHEAD* mbp, int error)
{
    putprintf(mbp, "(errno: %u, errtxt: %s)\r", errno, strerror(errno));
}

/**************************************************************************/
/* Interface zur Steuerung der Kernelfunktionen                           */
/*------------------------------------------------------------------------*/
void ccpkif(void)
{
  MBHEAD *mbp;

  char hostname[32];

  bzero(&hostname[0], sizeof(hostname));

  /* Funktionen im Sysop-Modus */
  if (issyso())
  {
      if (strnicmp((char *)clipoi, "CLEAR", 5) == 0)
      {
          ifip_clearstat();

          mbp = putals("Interface statistics cleared.\r");
          prompt(mbp);
          seteom(mbp);
          return;
      }

      if (strnicmp((char *)clipoi, "DOWN", 4) == 0)
      {
          mbp = putals("Interface ");

          if (kip_info.if_active == TRUE)
          {
              ifip_close(&kip_info);

              if (xaccess("KIF_DOWN.TNB", 0) == 0)
                 runbatch("KIF_DOWN.TNB");

              rt_drop(kip_info.kernel_ip, 32, FALSE);
              putstr("put to DOWN-state.\r", mbp);
          }
          else
              putstr("is DOWN.\r", mbp);

          prompt(mbp);
          seteom(mbp);
          return;
      }

      if (strnicmp((char *)clipoi, "UP", 2) == 0)
      {
          mbp = putals("Interface ");

          if (kip_info.if_available == FALSE)
          {
              putstr("is unavailable !!!\r", mbp);
              prompt(mbp);
              seteom(mbp);
              return;
          }

          if (kip_info.kernel_ip == 0L)
          {
              putstr("is not properly configured yet, set Kernel-IP with SETKIP !!!\r", mbp);
              prompt(mbp);
              seteom(mbp);
              return;
          }

          if (my_ip_addr == 0L)
          {
              putstr("is not properly configured yet, set Node-IP with IPA !!!\r", mbp);
              prompt(mbp);
              seteom(mbp);
              return;
          }

          if (kip_info.if_active == TRUE)
          {
              putstr("is already UP !\r", mbp);
              prompt(mbp);
              seteom(mbp);
              return;
          }

          if (ifip_setup(&kip_info) == FALSE)
              putstr("setup failed !!!\r", mbp);
          else
          {
              rt_add(kip_info.kernel_ip, 32, 0L, KERNEL_PORT, 0, 0, 0, FALSE);

              if (xaccess("KIF_UP.TNB", 0) == 0)
                 runbatch("KIF_UP.TNB");

              putstr("setup successfully\r", mbp);
          }

          prompt(mbp);
          seteom(mbp);
          return;
      }

      if (strnicmp((char *)clipoi, "INIT", 4) == 0)
      {
          if (kip_info.if_active == TRUE)
          {
              mbp = putals("Interface is UP, put it to DOWN-state before initializing !!!\r");
              prompt(mbp);
              seteom(mbp);
              return;
          }

          mbp = putals("Kernel-Interface initialization ");

          ifip_init(&kip_info);

          if (ifip_usable(&kip_info) == FALSE)
              putstr("FAILED, feature not available !!!\r", mbp);
          else
              putstr("successful\r", mbp);

          if (my_ip_addr == 0)
              putstr("WARNING: Node IP-Adress not yet set, set with IPA-Command !!!\r", mbp);

          prompt(mbp);
          seteom(mbp);
          return;
      }

      if (strnicmp((char *)clipoi, "STATUS", 5) == 0)
      {
          mbp = putals("Kernel-Interface status:\r");

          putstr("Kernel feature available : ", mbp);

          if (kip_info.if_available == FALSE)
              putstr("NO, not initialized or unavailable, try INIT.\r", mbp);
          else
          {
              putstr("yes, ",mbp);

              switch (kip_info.if_style)
              {
                  case 256: putstr("Kernel 2.4.x-style\r", mbp);
                            break;

                  default : putstr("Kernel 2.2.x-style\r", mbp);
              }
          }

          putstr("Kernel-Interface active  : ", mbp);

          if (kip_info.if_active == FALSE)
              putstr("no\r", mbp);
          else
              putstr("yes\r",mbp);

          putstr("IP-Adress of Kernel is   : ", mbp);

          if (kip_info.kernel_ip == 0)
              putstr("not defined\r", mbp);
          else
          {
              show_ip_addr(kip_info.kernel_ip, mbp);
              putchr('\r', mbp);
          }

          putstr("Bytes sent via Kernel    : ", mbp);
          putnum(kip_info.bytes_tx, mbp);

          putstr("\rBytes rcvd via Kernel    : ", mbp);
          putnum(kip_info.bytes_rx, mbp);

          putchr('\r', mbp);
          prompt(mbp);
          seteom(mbp);
          return;
      }

      if (strnicmp((char *)clipoi, "SETKIP", 6) == 0)
      {
          if (kip_info.if_active == TRUE)
          {
              mbp = putals("Interface is UP, put it DOWN before changing settings !!! \r");
              prompt(mbp);
              seteom(mbp);
              return;
          }

          nextspace(&clicnt, &clipoi);
          skipsp(&clicnt, &clipoi);

          sscanf(clipoi, "%s", hostname);

          mbp = putals("Kernel-IP ");

          if (ifip_kernelip(&kip_info, &hostname[0]) == FALSE)
              putstr("could not be set !!!\r", mbp);
          else
          {
              putstr("set to ", mbp);
              show_ip_addr(kip_info.kernel_ip, mbp);
              putchr('\r', mbp);
          }

          prompt(mbp);
          seteom(mbp);
          return;

      }

      mbp = putals("Usage: KERNELIF [INIT, CLEAR, SETKIP, UP, DOWN, STATUS]\r");
      prompt(mbp);
      seteom(mbp);
      return;
  }
  else
    invmsg();
}

/* Den Infoblock initialisieren */
static void ifip_init(struct kip *if_info)
{
    if_info->if_available = FALSE;
    if_info->if_style = 0;
    if_info->if_active = FALSE;
    if_info->kernel_ip = 0L;
    if_info->if_fd = -1;
    if_info->bytes_rx = 0L;
    if_info->bytes_tx = 0L;
}

/* Pruefen, ob notwendige Funktionen verfuegbar sind */
static BOOLEAN ifip_usable(struct kip *if_info)
{
    int fd = 0;
    int i = 0;

    char ifdev[14];

    if ((fd = open(IFIP, O_RDWR)) > 0)
    {
        if_info->if_available = TRUE;
        if_info->if_style = 256;
        close(fd);
        return TRUE;
    }

    for (i = 0; i < 255; i++)
    {
        sprintf(ifdev, "/dev/tun%d", i);
        if ((fd = open(ifdev, O_RDWR)) > 0)
        {
            if_info->if_available = TRUE;
            if_info->if_style = i;
            close(fd);
            return TRUE;
        }
    }

    if_info->if_available = FALSE;
    if_info->if_style = 0;
    return FALSE;
}

/* Den Link zum Kernel schliessen und im Infoblock als inaktiv vermerken */
static void ifip_close(struct kip *if_info)
{
    if ((if_info->if_active == TRUE) && (if_info->if_fd >= 0))
    {
        close(if_info->if_fd);
        if_info->if_fd = -1;
        if_info->if_active = FALSE;
    }
}

/* Das Interface kreieren */
BOOLEAN ifip_setup(struct kip *if_info)
{
    int fd = 0;
    int sd = 0;
    int err = 0;
    int i = 0;

    char dev[IFNAMSIZ];
    char ifdev[14];

    MBHEAD *mbp;

    struct ifreq ifr;

    bzero(&ifr, sizeof(ifr));

    if (if_info->if_style == 256)
    {
        sprintf(dev, "tnn");

        fd = open(IFIP, O_RDWR);

        if (fd <= 0)
        {
            close(fd);
            mbp = putals("Can't open device !!!\r");
            show_error(mbp, errno);
            seteom(mbp);
            return FALSE;
        }

        ifr.ifr_flags = (IFF_TUN | IFF_NO_PI);

        if ((err = ioctl(fd, TUNSETIFF, (void *) &ifr)) < 0 )
        {
            close(fd);
            mbp = putals("Can't set neccessary device flags !!!\r");
            show_error(mbp, errno);
            seteom(mbp);
            return FALSE;
        }

        bzero(&ifr, sizeof(ifr));
        ifr.ifr_flags = (IFF_TUN | IFF_NO_PI);
        strncpy(ifr.ifr_name, &dev[0], IFNAMSIZ);

        if ((err = ioctl(fd, TUNSETIFF, (void *) &ifr)) < 0 )
        {
            close(fd);
            mbp = putals("Can't set miscelleanous device flags, trying dynamic method...\r");
            seteom(mbp);

            fd = open(IFIP, O_RDWR);

            bzero(&ifr, sizeof(ifr));
            ifr.ifr_flags = (IFF_TUN | IFF_NO_PI);

            if (((err = ioctl(fd, TUNSETIFF, (void *) &ifr)) < 0 ) || (fd <= 0))
            {
                close(fd);
                mbp = putals("Can't set device flags dynamically !!!\r");
                show_error(mbp, errno);
                seteom(mbp);
                return FALSE;
            }

            mbp = putals("");
            putprintf(mbp, "Info: interface's name changed from '%s' to '%s' !!!\r", dev, ifr.ifr_name);
            seteom(mbp);

            strcpy(&dev[0], ifr.ifr_name);
        }
    }
    else
    {
        sprintf(ifdev, "/dev/tun%d", if_info->if_style);

        fd = open(ifdev, O_RDWR);

        if (fd <= 0)
        {
            close(fd);
            mbp = putals("Can't open device !!!\r");
            show_error(mbp, errno);
            seteom(mbp);
            return FALSE;
        }

        sprintf(dev, "tun%d", if_info->if_style);
    }

    bzero(&ifr, sizeof(ifr));

    sd = socket(AF_INET, SOCK_DGRAM, 0);

    if (sd <= 0)
    {
        close(sd);
        mbp = putals("Can't set up interface !!!\r");
        show_error(mbp, errno);
        seteom(mbp);
        return FALSE;
    }

    strcpy(ifr.ifr_name, &dev[0]);

    ifr.ifr_addr.sa_family = AF_INET;

    ifr.ifr_addr.sa_data[0] = 0;
    ifr.ifr_addr.sa_data[1] = 0;
    ifr.ifr_addr.sa_data[6] = 0;

    for (i = 0; i < 4; i++)
        ifr.ifr_addr.sa_data[5-i] = (if_info->kernel_ip >> 8 * i) & 0xff;

    if (ioctl(sd, SIOCSIFADDR, &ifr) < 0)
    {
        close(sd);
        mbp = putals("Can't set kernel-side ip-adress !!!\r");
        show_error(mbp, errno);
        seteom(mbp);
        return FALSE;
    }

    for (i = 0; i < 4; i++)
        ifr.ifr_addr.sa_data[5-i] = (my_ip_addr >> 8 * i) & 0xff;

    if (ioctl(sd, SIOCSIFDSTADDR, &ifr) < 0)
    {
        close(sd);
        mbp = putals("Can't set tnn-side ip-adress !!!\r");
        show_error(mbp, errno);
        seteom(mbp);
        return FALSE;
    }

    if ((err = ioctl(sd, SIOCGIFFLAGS, &ifr)) < 0)
    {
        close(sd);
        mbp = putals("Can't read interface flags !!!\r");
        show_error(mbp, errno);
        seteom(mbp);
        return FALSE;
    }

    ifr.ifr_flags &= IFF_NOARP;
    ifr.ifr_flags |= IFF_UP;
    ifr.ifr_flags |= IFF_RUNNING;

    if ((err = ioctl(sd, SIOCSIFFLAGS, &ifr)) < 0)
    {
        close(sd);
        mbp = putals("Can't set interface flags !!!\r");
        show_error(mbp, errno);
        seteom(mbp);
        return FALSE;
    }

    close(sd);

    strcpy(dev, ifr.ifr_name);

    if_info->if_active = TRUE;
    if_info->if_fd = fd;

    return TRUE;
}

/* Die Statistik des Infoblocks loeschen */
void ifip_clearstat(void)
{
    kip_info.bytes_rx = 0L;
    kip_info.bytes_tx = 0L;
}

/* IP des Kernels im Infoblock eintragen */
static BOOLEAN ifip_kernelip(struct kip *if_info, char *hostname)
{
    int i = 0;
    struct hostent *hp = NULL;

    if ((hp = gethostbyname(hostname)) == NULL)
        return FALSE;

    if_info->kernel_ip = 0L;

    for (i = 0; i < 4; i++)
       if_info->kernel_ip = (if_info->kernel_ip << 8) | hp->h_addr_list[0][i];

    return TRUE;
}

/* Statistik ausgeben */
void ifip_dispstat(MBHEAD *mbp)
{
    putstr("\rKernel-Interface statistics:", mbp);

    putstr("\rBytes received : ", mbp);
    putnum(kip_info.bytes_rx, mbp);

    putstr("\rBytes sent     : ", mbp);
    putnum(kip_info.bytes_tx, mbp);

    putstr("\r", mbp);
}

/* Interface aktiv ? Wenn ja wird der verwendete Descriptor gemeldet, wenn */
/* nicht wird -1 zurueckgemeldet */
int ifip_active(void)
{
    if (kip_info.if_active == TRUE)
         return (kip_info.if_fd);
    else return (-1);
}

/* Ein IP-Frame vom Kernel holen und in den internen Router einschleusen. */
/* Sollte nur aufgerufen werden, wenn der Descriptor lesbar ist. */
void ifip_frame_to_router(void)
{
    char buf[4096];
    int i = -1;
    int j = 0;

    MBHEAD *mbhd;

    if ((i = read(kip_info.if_fd, &buf[0], 4096)) <= 0)
    {
        /* irgendwas lief schief, wir machen dicht */
        close(kip_info.if_fd);
        kip_info.if_fd = -1;
        kip_info.if_active = FALSE;

        return;
    }

    kip_info.bytes_rx += (ULONG)i;

    (mbhd = (MBHEAD *)allocb(ALLOC_MBHEAD))->l2port = KERNEL_PORT;

    for (j = 0; j < i; j++)
      putchr(buf[j], mbhd);

    rwndmb(mbhd);

    relink((LEHEAD *)mbhd, (LEHEAD *)iprxfl.tail);
}

void ifip_frame_to_kernel(MBHEAD *mhbp)
{
    char buf[2048];
    int i = -1;
    int j = 0;

    /* falls das Interface nicht aktiv ist und noch Routing-Leichen da sind */
    /* die noch was auf dem Port abladen, einfach wegschmeissen */
    if (kip_info.if_active == FALSE)
    {
      dealmb(mhbp);
      return;
    }

    rwndmb(mhbp);

    while (mhbp->mbgc < mhbp->mbpc) /* solange Daten vorhanden sind */
        buf[j++] = getchr(mhbp);

    dealmb(mhbp);

    kip_info.bytes_tx += (ULONG)j;

    if ((i = write(kip_info.if_fd, &buf[0], j)) <= 0)
    {
        /* irgendwas lief schief, wir machen dicht */
        close(kip_info.if_fd);
        kip_info.if_fd = -1;
        kip_info.if_active = FALSE;
    }
}

/* Die Konfiguration des Kernelinterfaces fuer parms.tnb dumpen */
void dump_kernel(MBHEAD *mbp)
{
  putstr(";\r; Kernel Interface Configuration\r;\r", mbp);

  /* nur bei AKTIVIERTEM und LAUFENDEN Interface und wenn dem Kernel */
  /* eine IP-Adresse zugewiesen worden ist                           */
  if ((kip_info.if_active == FALSE) ||  (kip_info.kernel_ip == 0))
  {
    putstr("; (no information dumped because interface was not running\r", mbp);
    putstr(";  or not properly configured)\r", mbp);
    return;
  }

  putstr("KERN INIT\r", mbp);

  /* die IP-Adresse des Kernels dumpen */
  putstr("KERN SETKIP ", mbp);
  show_ip_addr(kip_info.kernel_ip, mbp);
  putstr("\r", mbp);

  putstr("KERN UP\r", mbp);
}

/*****************************************************************************/
/* ab hier Funktionen fuer Kernel AX.25                                      */
/*****************************************************************************/

/* an ein Kernel-AX.25-Interface anhaengen */
BOOLEAN ifax_setup(DEVICE *l1pp)
{
  struct ifreq ifr;
  struct sockaddr sa;

  MBHEAD *mbp;

  /* schon laufende und nicht passende Ports fassen wir nicht an */
  if (   (l1pp->port_active != TRUE)
      || ((l1pp->kisstype != KISS_KAX25) && (l1pp->kisstype != KISS_KAX25KJD))
      || (l1pp->kisslink > -1))
    return (FALSE);

  /* Socket anlegen */
  if ((l1pp->kisslink = socket(AF_PACKET, SOCK_PACKET, htons(ETH_P_ALL))) < 0)
  {
    mbp = putals("Can't create socket !\r");
    show_error(mbp, errno);
    seteom(mbp);

    return (FALSE);
  }

  /* Interfacestruktur bereinigen und Interface auswaehlen */
  bzero(&ifr, sizeof(ifr));
  strcpy(ifr.ifr_name, l1pp->device);

  /* vom Kernel die aktuellen Interfaceeinstellungen holen */
  if (ioctl(l1pp->kisslink, SIOCGIFFLAGS, &ifr) < 0)
  {
    /* dieses Interface gab es aber nicht */
    mbp = putals("Can't get current interface flags from kernel !\r");
    show_error(mbp, errno);
    seteom(mbp);

    close(l1pp->kisslink);
    return (FALSE);
  }

  /* Flags des Interfaces sichern */
  l1pp->oldifparms = ifr.ifr_flags;

  /* sicherheitshalber Interface aktivieren */
  ifr.ifr_flags |= IFF_UP;

  ifr.ifr_flags |= IFF_PROMISC;

  /* Aenderungen zum Interface bringen */
  if (ioctl(l1pp->kisslink, SIOCSIFFLAGS, &ifr) < 0)
  {
    mbp = putals("Can't set interface flags !\r");
    show_error(mbp, errno);
    seteom(mbp);

    ifax_close(l1pp);
    return (FALSE);
  }

  /* Socketstruktur loeschen und danach fuellen */
  bzero(&sa, sizeof(sa));
  strcpy(sa.sa_data, l1pp->device);
  sa.sa_family = AF_INET;

  /* Interface binden */
  if (bind(l1pp->kisslink, &sa, sizeof(struct sockaddr)) < 0)
  {
    mbp = putals("Can't bind interface !\r");
    show_error(mbp, errno);
    seteom(mbp);

    ifax_close(l1pp);
    return (FALSE);
  }

  return (TRUE);
}

/* Interface schliessen */
void ifax_close(DEVICE *l1pp)
{
  struct ifreq ifr;

  /* alten Zustand wiederherstellen */
  ifr.ifr_flags = l1pp->oldifparms;
  ioctl(l1pp->kisslink, SIOCSIFFLAGS, &ifr);

  /* Schliessen und austragen */
  close(l1pp->kisslink);
  l1pp->kisslink = -1;
}

/* Portinfo-String fuer den PORT-Befehl */
void ifax_hwstr(int port, MBHEAD *mbp)
{
  putprintf(mbp, " Interface %s", &(l1port[l1ptab[port]].device[0]));
}

/* Daten vom Interface abholen, vorher muss mit select geprueft werden */
/* ob auch Daten da sind (Stichwort select() ) */
void ifax_rx(int fd)
{
  int i = 0;
  int j = 0;
  int port = 0;
  char buf[2048];

  MBHEAD *mbhd;

  for (port = 0; port < L2PNUM; port++)
  {
    /* nur aktive Kernelports und davon den richtigen mit dem aktiven fd */
    if (   (l1port[l1ptab[port]].port_active != TRUE)
        || (l1port[l1ptab[port]].kisstype < KISS_KAX25)
        || (l1port[l1ptab[port]].kisstype > KISS_KAX25KJD)
        || (l1port[l1ptab[port]].kisslink != fd))
      continue;

    bzero(buf, sizeof(buf));
    errno = 0;

    i = recvfrom(l1port[l1ptab[port]].kisslink, &buf[0], 2048, 0, NULL, 0);

    /* Fehler abfangen */
    if (i <= 0)
    {
      if (i < 0)
      {
        /* es konnte nicht richtig gesendet werden, Port schliessen */
        l1detach(port);
      }
      return;
    }

    if (l1port[l1ptab[port]].kisstype == KISS_KAX25) // normaler Kernelstack
    {
      j = 1;
      /* Sortieren was da gekommen ist, Parameteraenderungen uebernehmen */
      switch (buf[0] & 0x0F)
      {
        case 0 : break; /* normale Frames */
        case PARAM_TXDELAY:
          portpar[port].txdelay = (int)buf[1];
          autopar(port);
          return;
        case PARAM_SLOTTIME:
          portpar[port].slottime = (int)buf[1];
          autopar(port);
          return;
        case PARAM_PERSIST:
          portpar[port].persistance = (int)buf[1];
          autopar(port);
          return;
        case PARAM_FULLDUP:
          if (buf[1])
            portpar[port].l1mode = portpar[port].l1mode & MODE_d;
          else
            portpar[port].l1mode = portpar[port].l1mode & ~MODE_d;
          autopar(port);
          return;
        default : return; /* alles unbekannte wird nicht weitergereicht */
      }
    }

    /* Puffer besorgen */
    (mbhd = (MBHEAD *)allocb(ALLOC_MBHEAD))->l2port = port;

    /* Puffer kopieren */
    for (; j < i; j++)
      putchr(buf[j], mbhd);

    /* in Empfangsliste einhaengen */
    relink((LEHEAD *)mbhd, (LEHEAD *)rxfl.tail);
  }
}

/* Sendedaten an das Interface schicken, keine Parameter */
void ifax_tx(void)
{
  struct sockaddr to;
  char buf[2048];
  int count = 1;
  int i = 0;
  int port;

  MBHEAD *txfhdl;
  LHEAD  *l2flp;

  l2flp = txl2fl;

  for (port = 0; port < L2PNUM; l2flp++, port++)
  {
    if ((l1port[l1ptab[port]].port_active != TRUE) ||
        (l1port[l1ptab[port]].kisstype < KISS_KAX25) ||
        (l1port[l1ptab[port]].kisstype > KISS_KAX25KJD) ||
        (l1port[l1ptab[port]].kisslink < 0))
      continue;

    if (portenabled(port))
    {
      if (kick[port])                   /* was zum senden...            */
      {
        if (l2flp->head == l2flp)
        {
          kick[port] = FALSE;
          continue;
        }

        bzero(&buf[0], sizeof(buf));

        /* KJD-Stack */
        if (l1port[l1ptab[port]].kisstype == KISS_KAX25KJD)
          count = 0;

        ulink((LEHEAD *)(txfhdl = (MBHEAD *) l2flp->head));/*Zeiger holen*/
        while (txfhdl->mbgc < txfhdl->mbpc) /* solange Daten vorhanden sind */
          buf[count++] = getchr(txfhdl);

        relink((LEHEAD *)txfhdl,       /* als gesendet betrachten und in */
              (LEHEAD *)stfl.tail);    /* die gesendet Liste umhaengen   */

        kick[port] = ((LHEAD *)l2flp->head != l2flp);

        bzero(&to, sizeof(to));
        errno = 0;

        /* Interface setzen auf dem dieses Paket gesendet werden soll */
        strcpy(to.sa_data, l1port[l1ptab[port]].device);

        /* Puffer senden */
        i = sendto(l1port[l1ptab[port]].kisslink, &buf[0], count, 0, &to, sizeof(to));

        /* Fehler abfangen */
        if (i < 0)
          l1detach(port);
      }
    }
    else
      while (l2flp->head != l2flp)
        relink(ulink((LEHEAD *) l2flp->head), (LEHEAD *)stfl.tail);
  }
}

/* Setzt die Parameter (Slottime, TX-Delay usw.) */
void ifax_param(int port)
{
  struct sockaddr to;
  unsigned char buf[2];
  int count;
  int i;

  MBHEAD *mbp;

  /* Der KJD-Stack darf hier nicht rein, deshalb muss sein Kenncode */
  /* bei der folgenden if-Abfrage fehlen !!!                        */
  if (   (l1port[l1ptab[port]].port_active != TRUE)
      || (l1port[l1ptab[port]].kisstype != KISS_KAX25)
      || (l1port[l1ptab[port]].kisslink < 0)
     )
       return;

  /* Interface waehlen */
  bzero(&to, sizeof(to));
  strcpy(to.sa_data, l1port[l1ptab[port]].device);

  /* alle Befehle sind zwei Bytes lang, globale error-Variable loeschen */
  count = 2;
  errno = 0;

  /* TX-Delay senden */
  buf[0] = PARAM_TXDELAY;
  buf[1] = portpar[port].txdelay;

  i = sendto(l1port[l1ptab[port]].kisslink, &buf[0], count, 0, &to, sizeof(to));

  if (i < 0)
  {
    mbp = putals("");
    putprintf(mbp, "Can't set txdelay on interface %s !\r", l1port[l1ptab[port]].device );
    show_error(mbp, errno);
    putprintf(mbp, "Port has been deactivated !!!\r");
    seteom(mbp);

    l1detach(port);
    return;
  }

  /* Slottime senden */
  buf[0] = PARAM_SLOTTIME;
  buf[1] = portpar[port].slottime;

  i = sendto(l1port[l1ptab[port]].kisslink, &buf[0], count, 0, &to, sizeof(to));

  if (i < 0)
  {
    mbp = putals("");
    putprintf(mbp, "Can't set slottime on interface %s !\r", l1port[l1ptab[port]].device );
    show_error(mbp, errno);
    putprintf(mbp, "Port has been deactivated !!!\r");
    seteom(mbp);

    l1detach(port);
    return;
  }

  /* Persistance senden */
  buf[0] = PARAM_PERSIST;
  if (dama(port))
    buf[1] = 255;
  else
    buf[1] = portpar[port].persistance;

  i = sendto(l1port[l1ptab[port]].kisslink, &buf[0], count, 0, &to, sizeof(to));

  if (i < 0)
  {
    mbp = putals("");
    putprintf(mbp, "Can't set persistence on interface %s !\r", l1port[l1ptab[port]].device );
    show_error(mbp, errno);
    putprintf(mbp, "Port has been deactivated !!!\r");
    seteom(mbp);

    l1detach(port);
    return;
  }

  /* Vollduplex setzen */
  buf[0] = PARAM_FULLDUP;
  if (fullduplex(port))
    buf[1] = 1;
  else
    buf[1] = 0;

  i = sendto(l1port[l1ptab[port]].kisslink, &buf[0], count, 0, &to, sizeof(to));

  if (i < 0)
  {
    mbp = putals("");
    putprintf(mbp, "Can't set half/full-duplex on interface %s !\r", l1port[l1ptab[port]].device );
    show_error(mbp, errno);
    putprintf(mbp, "Port has been deactivated !!!\r");
    seteom(mbp);

    l1detach(port);
    return;
  }
}

/* Level 1 Kontrolle */
void ifax_l1ctl(int req)
{
  switch (req)
  {
    case L1CCMD : update_parms = TRUE; break; /* Portparameter setzen */
    default     : break;                /* den Rest koennen wir nicht */
  }
}

/* generelle Funktion, sendet ausstehende Frames und bringt ggf. Parameter- */
/* aenderungen zum Interface usw.*/
void ifax_housekeeping(void)
{
  register int port;

  ifax_tx();

  if (update_parms)
  {
    for (port = 0; port < L2PNUM; ++port)
    {
      if (l1port[l1ptab[port]].kisstype != KISS_KAX25)
        continue;

      if (commandflag[port])
      {
        ifax_param(port);
        commandflag[port] = FALSE;
      }
    }
    update_parms = FALSE;
  }
}
#endif

/* End of os/linux/kernelif.c */
