/************************************************************************/
/*                                                                      */
/*    *****                       *****                                 */
/*      *****                   *****                                   */
/*        *****               *****                                     */
/*          *****           *****                                       */
/*  ***************       ***************                               */
/*  *****************   *****************                               */
/*  ***************       ***************                               */
/*          *****           *****           TheNetNode                  */
/*        *****               *****         Portable                    */
/*      *****                   *****       Network                     */
/*    *****                       *****     Software                    */
/*                                                                      */
/* File src/l2misc.c (maintained by: DF6LN)                             */
/*                                                                      */
/* This file is part of "TheNetNode" - Software Package                 */
/*                                                                      */
/* Copyright (C) 1998 - 2000 NORD><LINK e.V. Braunschweig               */
/*                                                                      */
/* This program is free software; you can redistribute it and/or modify */
/* it under the terms of the NORD><LINK ALAS (Allgemeine Lizenz fuer    */
/* Amateurfunk Software) as published by Hans Georg Giese (DF2AU)       */
/* on 13/Oct/1992; either version 1, or (at your option) any later      */
/* version.                                                             */
/*                                                                      */
/* This program is distributed WITHOUT ANY WARRANTY only for further    */
/* development and learning purposes. See the ALAS (Allgemeine Lizenz   */
/* fuer Amateurfunk Software).                                          */
/*                                                                      */
/* You should have received a copy of the NORD><LINK ALAS (Allgemeine   */
/* Lizenz fuer Amateurfunk Software) along with this program; if not,   */
/* write to NORD><LINK e.V., Hinter dem Berge 5, D-38108 Braunschweig   */
/*                                                                      */
/* Dieses Programm ist PUBLIC DOMAIN, mit den Einschraenkungen durch    */
/* die ALAS (Allgemeine Lizenz fuer Amateurfunk Software), entweder     */
/* Version 1, veroeffentlicht von Hans Georg Giese (DF2AU),             */
/* am 13.Oct.1992, oder (wenn gewuenscht) jede spaetere Version.        */
/*                                                                      */
/* Dieses Programm wird unter Haftungsausschluss vertrieben, aus-       */
/* schliesslich fuer Weiterentwicklungs- und Lehrzwecke. Naeheres       */
/* koennen Sie der ALAS (Allgemeine Lizenz fuer Amateurfunk Software)   */
/* entnehmen.                                                           */
/*                                                                      */
/* Sollte dieser Software keine ALAS (Allgemeine Lizenz fuer Amateur-   */
/* funk Software) beigelegen haben, wenden Sie sich bitte an            */
/* NORD><LINK e.V., Hinter dem Berge 5, D-38108 Braunschweig            */
/*                                                                      */
/************************************************************************/

#include "tnn.h"

static void l2rest(void);
static void inilbl(void);

/*----------------------------------------------------------------------*/
/*                                                                      */
/* "level 2"                                                            */
/*                                                                      */
/* Der Level 2. Es werden alle Level-2-internen Aktionen ausgefuehrt    */
/* und Meldungen an hoehere Level weitergegeben (Informationstransfer   */
/* von/zum Level 2 und Kommandos an den Level 2 geschehen von ausser-   */
/* halb).                                                               */
/*                                                                      */
/* Der Level 2 laeuft wie folgt ab :                                    */
/*                                                                      */
/*  - Aufruf von l2init()                                               */
/*                                                                      */
/*  - zyklisches Aufrufen von l2()                                      */
/*                                                                      */
/*  Statusaenderungen im Level 2 (Connects, Disconnects, Failures, usw) */
/*  werden hoeheren Leveln vom Level 2 aus ueber                        */
/*                                                                      */
/*   l2tolx(<status>)  ->  l2tol3(<status>), l2tol7(<status>,lnkpoi,2)  */
/*                                                                      */
/*  mitgeteilt.                                                         */
/*                                                                      */
/*  Ein Connectwunsch wird dem Level 2 ueber das Besetzen eines leeren  */
/*  Linkblocks mit Quell- und Ziel- sowie Digicalls und Aufrufen von    */
/*  newlnk() mitgeteilt (lnkpoi zeigt auf Linkblock !).                 */
/*  Ein newlnk() auf einen bestehenden Link erzeugt einen Link Reset.   */
/*                                                                      */
/*  Ein Disconnectwunsch (oder bei mehrmaligem Aufruf der sofortige     */
/*  Disconnect) wird ueber das Setzen von lnkpoi auf den jeweiligen     */
/*  Linkblock und Aufruf von dsclnk() erreicht.                         */
/*                                                                      */
/*  Der Informationstransfer zum Level 2 geschieht von aussen durch     */
/*  Aufruf von itolnk(...), vom Level 2 durch itolx(..), welches dann   */
/*  fmlink() aus dem hoeheren Level aufruft.                            */
/*                                                                      */
/*  Ueber sdui(..) koennen unproto-Pakete (UI-Frames) gesendet werden.  */
/*                                                                      */
/*  Level-3-Pakete (Level-3-UI-Pakete oder Infopakete in Sequenz eines  */
/*  Level-2-3-Links) werden ausgefiltert und in die Level-3-Frameliste  */
/*  eingehaengt.                                                        */
/*                                                                      */
/*----------------------------------------------------------------------*/

/* Die Reihenfolge der Bearbeitung ist hier entscheidend. Zuerst werden */
/* die empfangenen Frames verarbeitet, dann werden I-Frames erzeugt und */
/* erst dann wird ueber T1/T2 entschieden.                              */

void
l2(void)
{
  clrstfl();
  l2rx();
  l2tx();
  l2timr();
  l2rest();
}

/*----------------------------------------------------------------------*/
void
l2init(void)
{
  WORD i;

  inithd(&rxfl);
  inithd(&stfl);
  inithd(&trfl);
  inithd(&freel);
  inithd(&l2frel);

  init_buffers();

  for (i = 0; i < L2PNUM; ++i)
  {
    inithd(&txl2fl[i]);
    inithd(&l2actl[i]);
  }

  for (i = 0, lnkpoi = lnktbl; i < LINKNMBR; ++i, ++lnkpoi)
  {
    lnkpoi->state = L2SDSCED;
    lnkpoi->rcvd = lnkpoi->tosend = 0;
    lnkpoi->tmbp = NULL;
    lnkpoi->zael = 1;                                   /* MultiConnect */
    inithd(&lnkpoi->rcvdil);
    inithd(&lnkpoi->sendil);
    inilbl();
    reslnk();
    relink((LEHEAD *)lnkpoi, (LEHEAD *)l2frel.tail);
  }

  iniDAMA();
  for (i = 0; i < L2CALEN; i++)
    l2als[i] = toupper(alias[i]);
}

void
autopar(int port)
{
  PORTINFO *p = portpar + port;
  UWORD     baud;               /* Baudrate/100 auf dem Port            */
  UWORD     P;                  /* Persistance                          */
  UWORD     W;                  /* Slottime                             */
  UWORD     IRTT;               /* Initial RTT                          */
  UWORD     T2;                 /* Timer2                               */
  UWORD     N2;                 /* Retry                                */
  BOOLEAN   l1update;

/* T2 wird berechnet als die Laenge der Aussendung eines Frames bei     */
/* der eingestellten Baudrate + 10%.                                    */
  baud = p->speed;

  T2 = 1;                               /* sehr schneller Link          */
  if ((baud != 0) && (baud < 2888))
    T2 = 2888 / baud;                   /* sonst adaptiv nach Baudrate  */
  if (dama(port))
    T2 <<= 1;                           /* *2 bei DAMA nach DG3AAH      */

/* Der IRTT wird auf T2*2 festgelegt, dies sollte immer
 * ausreichen, damit eine Antwort kommt. Es wird noch der TXDELAY
 * dazugerechnet, in der Annahme die Gegenstation habe etwa einen
 * gleich schnellen Transceiver.
 */
  IRTT = (T2 + p->txdelay) * 2;
  IRTT = max(10, IRTT);

/* Der Retry ist DEF_N2 bei DUPLEX/CSMA und DEF_N2/2 bei DAMA.          */
  N2 = DEF_N2;
  if (dama(port))
    N2 /= 2;

/* P-PERSISTENCE
 * Bei DAMA und VOLLDUPLEX wird P immer auf 255 gesetzt.
 * Bei einem normalen Einstieg ohne DAMA oder Halb-Duplex
 * verwenden wir defaultmaessig P=160
 */
  P = 255;
  if (!fullduplex(port) && !dama(port))
    P = 160;                                    /* Idee DB2OS und DG9FU */

/* Als Slottime wird der aktuelle TX-Delay Wert genommen.               */
  W = p->txdelay;
  if (W == 0)
    W++;

  l1update = (P != p->persistance)
    || (W != p->slottime);

  p->persistance = P;                           /* Parameter neu setzen */
  p->slottime = W;
  p->IRTT = IRTT;
  p->T2 = T2;
  p->retry = N2;

  if (l1update)                         /* neue L1-Parameter setzen     */
    l1ctl(L1CCMD, port);
}

void
autopers(int port)
{
#ifdef DAMASLAVE
   PORTINFO *p = portpar + port;

   if (damaslaveon(port))               /* wenn DAMA erkannt, dann      */
   {
     p->persistance = 255;              /* sofort antworten             */
     p->slottime = 0;                   /* keine Wartezeit              */
     l1ctl(L1CCMD, port);               /* Aenderungen zum TNC bringen  */
   }
#endif

#if 0                                           /* DB2OS will es so ... */
  PORTINFO *p = portpar + port;
  UWORD     P;                                  /* Persistance          */
  BOOLEAN   l1update;

/* Persistance wird berechnet als 255-(Anzahl der User)*10. Bei         */
/* Vollduplex und DAMA wird die Persistance immer auf 255 gesetzt.      */
  P = 255;
  if ((p->nmbstn > 0) && !fullduplex(port) && !dama(port))
    if (p->nmbstn <= 15)
      P = 255 - (p->nmbstn * 10);
    else
      P = 100;

  l1update = (P != p->persistance);

  p->persistance = P;                   /* Parameter neu setzen         */

  if (l1update)                         /* neue L1-Parameter setzen     */
    l1ctl(L1CCMD, port);
#endif
}

/* Ist ein Port gerade aktiv ? (Blockierung des Senders und der Timer   */
BOOLEAN
busy(int port)
{
  if (fullduplex(port))
    return ((iscd(port) & (RXBFLAG | TXBFLAG)) != 0);
  else
    return ((iscd(port) & (DCDFLAG | PTTFLAG | RXBFLAG | TXBFLAG)) != 0);
}

/************************************************************************\
*                                                                        *
* "level 2 rest"                                                         *
*                                                                        *
* Muellbufferliste frei machen (aus Interruptroutinen entstandener       *
* Muell, der besser ausserhalb der Interrupts deallokiert wird aus       *
* Zeitgruenden).                                                         *
*                                                                        *
\************************************************************************/
static void
l2rest(void)
{
  dealml((LEHEAD *)&trfl);      /* Muellbufferliste frei machen         */
}

void
l2profiler(void)
{
#ifdef L2PROFILER
  MBHEAD *mbp;

  if (dmagic == MAGIC_L2PROFILE)
  {                                     /* im Level 2 ordentlich Krach  */
    while (   lnkpoi->tosend < 10       /* machen                       */
           && !(lnkpoi->flag & (L2FDSLE | L2FDIMM)))
    {
      mbp = (MBHEAD *)allocb(ALLOC_MBHEAD);
      while (mbp->mbpc < 256)
        putchr(' ', mbp);
      rwndmb(mbp);
      i3tolnk(0x12, lnkpoi, mbp);
    }
  }
#endif
}

/************************************************************************/
/*                                                                      */
/* "is to me"                                                           */
/*                                                                      */
/* Ist das uebergebene Rufzeichen unser myid oder alias? Der Alias kann */
/* mit beliebiger SSID connected werden.                                */
/*                                                                      */
/************************************************************************/
BOOLEAN
istome(const char *id)
{
  return (   cmpid(myid, id)            /* Stimmt Rufzeichen?           */
          || cmpcal(l2als, id));        /* oder ALIAS?                  */
}

/************************************************************************/
/*                                                                      */
/* "is to me or via me?"                                                */
/*                                                                      */
/* Pruefen, ob wir als naechster an der Reihe sind mit Digipeaten. Dazu */
/* schauen wir uns das erste Call an, das kein gesetztes Hbit hat, dass */
/* muessen wir selber sein oder wir schmeissen das Frame weg,           */
/* andernfalls duerfen wir das Frame auswerten.                         */
/*                                                                      */
/************************************************************************/
int
istomev(void)
{
  char *viap;

  for (viap = rxfhdr + L2ILEN; *viap; viap += L2IDLEN)
    if (!(viap[L2IDLEN - 1] & L2CH))
      return (istome(viap)                      /* Verbindung via uns?  */
              ? 2 : 0);
/* hier angekommen haben entweder alle schon gedigipeated oder es       */
/* gibt keine via's, auf jeden Fall duerfen wir den Link nehmen, wenn   */
/* er direkt an uns gerichtet ist.                                      */
  return (istome(rxfhdr) ? 1 : 0);
}

/************************************************************************\
*                                                                        *
* "new link"                                                             *
*                                                                        *
* Link (lnkpoi) neu aufbauen.                                            *
*                                                                        *
\************************************************************************/
void
newlnk(void)
{
  reslnk();                     /* Sequenzvars/Timer ruecksetzen        */
  setiSRTT();                   /* RTT-Timer neu starten                */
  l2stma(stbl19);               /* LOCAL START COMMAND                  */
}

/************************************************************************\
*                                                                        *
* "disconnect link"                                                      *
*                                                                        *
* Disconnect-Wunsch an aktuellen Link (lnkpoi) :                         *
*                                                                        *
*   Linkstatus "Disconnected"                                            *
*   -> Ax.25-Parameter "frisch"                                          *
*                                                                        *
*   Linkstatus "Link Setup" oder "Disconnect Request"                    *
*   -> Link NICHT aufloesen, nur Flag vormerken fuer l2rest              *
*                                                                        *
*   sonst                                                                *
*   -> Empfangsinfoframeliste loeschen, Linkstatus bleibt, Flag "nach    *
*      Loswerden aller zu sendenden Infoframes disconnecten" setzen      *
*                                                                        *
\************************************************************************/
void
dsclnk(void)
{
  WORD lstate;

  if ((lstate = lnkpoi->state) == L2SDSCED)     /* Disced, nur          */
    inilbl();                                   /* AX-Pars neu          */
  else
  {
    dealml((LEHEAD *)&lnkpoi->rcvdil);          /* RX-Infoframe-        */
    lnkpoi->rcvd = 0;                           /* loeschen und         */

    if (   lstate == L2SLKSUP                   /* Linksetup oder       */
        || lstate == L2SDSCRQ                   /* Discreq,             */
        || lstate == L2SHTH                     /* oder wartet          */
        || (lnkpoi->flag & L2FDSLE))
      lnkpoi->flag |= L2FDIMM;                  /* sofort weg           */
    else
      lnkpoi->flag |= L2FDSLE;                  /* wenn alles raus      */
  }
}

/************************************************************************\
*                                                                        *
* "initialize link"                                                      *
*                                                                        *
* Aktuellen Linkblock (lnkpoi) initialisieren. Sequenzvariablen und      *
* Timer initialisieren, Quellcall/Zielcall/via-Liste/ Port setzen aus    *
* der txf...-Liste.                                                      *
*                                                                        *
\************************************************************************/
void
inilnk(void)
{
  reslnk();                                /* Sequenzvars/Timer init.   */
  cpyid(lnkpoi->srcid, txfhdr + L2IDLEN);  /* Quellcall                 */
  cpyid(lnkpoi->dstid, txfhdr);            /* Zielcall                  */
  cpyidl(lnkpoi->viaidl, txfhdr + L2ILEN); /* via-Liste                 */
  lnkpoi->liport = txfprt;                 /* Port                      */
  setiSRTT();                              /* RTT                       */
  lnkpoi->pollcnt = 0;                     /* Anzahl verbotener Polls   */
                                           /* des DAMA-Users            */
  lnkpoi->noatou = ininat;                 /* Timeout initialisieren    */
  lnkpoi->maxframe = portpar[lnkpoi->liport].maxframe;
}

/************************************************************************\
*                                                                        *
* "clear link"                                                           *
*                                                                        *
* Aktuellen Link (lnkpoi) aufloesen. Alle Sequenzvariablen und Timer     *
* zuruecksetzen, Sende- und Empfangsinfoframelise loeschen, Linkblock    *
* neu mit AX.25-Parametern besetzen.                                     *
*                                                                        *
\************************************************************************/
void
clrlnk(void)
{
  reslnk();                           /* Sequenzvars/Timer ruecksetzen  */
  dealml((LEHEAD *)&lnkpoi->rcvdil);  /* Empfangsinfoliste loeschen     */
  dealml((LEHEAD *)&lnkpoi->sendil);  /* Sendeinfoliste loeschen        */
  dealml((LEHEAD *)&lnkpoi->damail);  /* DAMA-Puffer loeschen           */
  lnkpoi->rcvd = lnkpoi->tosend = 0;  /* entsprechende Zaehler loeschen */
  if (lnkpoi->tmbp != NULL)           /* RX-Fragmente auch              */
  {
    dealmb(lnkpoi->tmbp);
    lnkpoi->tmbp = NULL;
  }
  inilbl();                           /* Linkblock "frisch"             */
}

/************************************************************************\
*                                                                        *
* "reset link"                                                           *
*                                                                        *
* Aktuellen Link (lnkpoi) zuruecksetzen. Alle Sequenzvariablen und Timer *
* initialisieren.                                                        *
*                                                                        *
\************************************************************************/
void
reslnk(void)
{
  lnkpoi->VS =
    lnkpoi->VR =
    lnkpoi->ltxdNR =
    lnkpoi->lrxdNR =
    lnkpoi->RTT =
    lnkpoi->flag = 0;
  resptc(g_uid(lnkpoi, L2_USER));
  clrDAMA();
  lnkpoi->SRTT = portpar[lnkpoi->liport].IRTT;
  clrT1();
  clrT2();
}

/************************************************************************\
*                                                                        *
* "initialize link block"                                                *
*                                                                        *
* Aktuellen Linkblock (lnkpoi) initialisieren.                           *
*                                                                        *
\************************************************************************/
static void
inilbl(void)
{
  lnkpoi->liport = 0;
  clrDAMA();
}

/************************************************************************\
*                                                                        *
* "acknowledge link"                                                     *
*                                                                        *
* Diese Funktion wird vom L7 aufgerufen, um eine eingehende HTH-         *
* Verbindung zu bestaetigen. Es wird lediglich ein Flag gesetzt, in      *
* l2rest wird dann die eigentliche Reaktion ausgeloest.                  *
*                                                                        *
\************************************************************************/
void
acklnk(LNKBLK *lp)
{
  if (lp->state == L2SHTH)
    lp->flag |= L2FACKHTH;
}

/************************************************************************\
*                                                                        *
* "reject link"                                                          *
*                                                                        *
* Diese Funktion wird vom L7 aufgerufen, um eine eingehende HTH-         *
* Verbindung abzulehnen (Partner ist Busy).                              *
*                                                                        *
\************************************************************************/
void
rejlnk(LNKBLK *lp)
{
  if (lp->state == L2SHTH)
    lp->flag |= L2FREJHTH;
}

/************************************************************************\
*                                                                        *
* "get link"                                                             *
*                                                                        *
* Einen Link anhand port, srcid, dstid und vial suchen und liefern. Wenn *
* wir noch keinen Link haben, dann einen neuen (leeren) liefern.         *
*                                                                        *
\************************************************************************/
LNKBLK *
getlnk(UBYTE liport, char *srcid, char *dstid, char *viaidl)
{
  LHEAD  *llp;
  LNKBLK *lp;
  char   *p;

  llp = &l2actl[liport];                /* auf dem entsprechenden Port  */
  for (lp = (LNKBLK *)llp->tail;
       lp != (LNKBLK *)llp;             /* alle Links des Ports pruefen */
       lp = lp->prev)
  {
    if (cmpid(lp->srcid, srcid))
      if (cmpid(lp->dstid, dstid)
          && cmpidl(lp->viaidl, viaidl))
      {
        ulink((LEHEAD *)lp);
        relink((LEHEAD *)lp, (LEHEAD *)llp->tail);
        return (lp);            /* wir haben einen Link gefunden        */
      }
  }

  if ((LHEAD *)l2frel.head == &l2frel)
    return (NULL);              /* wir haben keinen freien Link         */

  if (   (!fvalca(dstid))
      || (nmbfre < 78))         /* falsches Call oder kein Speicher?    */
    return (NULL);

  lp = (LNKBLK *)l2frel.head;
  cpyid(lp->srcid, srcid);      /* Werte setzen                         */
  cpyid(lp->dstid, dstid);
  cpyidl(lp->viaidl, viaidl);
  lp->liport = liport;
/* In den Linkblock wird ein Zeiger auf das Rufzeichen abgelegt, das in */
/* Wirklichkeit der Ansprechpartner dieses Linkes ist. Es ist das Erste */
/* Rufzeichen im via-Feld ohne H-Bit oder das Ziel-Rufzeichen selbst.   */
  for (p = lp->viaidl; *p; p += L2IDLEN)
    if ((p[L2IDLEN - 1] & L2CH) == 0)
      break;
  lp->realid = *p ? p : lp->dstid;
  return (lp);                          /* leeren Block liefern         */
}

/************************************************************************/
/*                                                                      */
/* Maxframe-Automatik: Fuer den angegebenen Link wird Maxframe um den   */
/* Wert dif geaendert. Dabei wird der Bereich 1 .. Port-Maxframe        */
/* eingehalten. Zum Vergroessern des Link-Maxframe muss zweimal in      */
/* Folge alles bestaetigt sein.                                         */
/*                                                                      */
/************************************************************************/
void
change_maxframe(LNKBLK *link, int dif)
{
  char  notify_call1[10];
  char  notify_call2[10];
  int   old= link->maxframe;
  int   new;
  int   port_max= portpar[link->liport].maxframe;

  if (dif > 0)                          /* soll vergroessert werden?    */
  {
    if (old == port_max)                /* schon Maximalwert, den Rest  */
      return;                           /* kann man sich sparen         */

    if (link->flag & L2FCMDEL)          /* 2. Durchgang in Folge?       */
      link->flag &= ~L2FCMDEL;
    else
    {
      link->flag |= L2FCMDEL;           /* 1. Durchgang - nix aendern   */
      dif = 0;
    }
  }
  else
    link->flag &= ~L2FCMDEL;

  new = old + dif;
  if (new < 1)
    new = 1;
  if (new > port_max)
    new = port_max;

  if ((new != old) | (link->flag & L2FCMDEL))
  {
    link->maxframe = new;
    call2str(notify_call1, link->srcid);
    call2str(notify_call2, link->dstid);
    notify(1, "Max %s - %s %d->%d",
           notify_call1, notify_call2, old, new);
  }
}

/* End of src/l2misc.c */
