/************************************************************************/
/*                                                                      */
/*    *****                       *****                                 */
/*      *****                   *****                                   */
/*        *****               *****                                     */
/*          *****           *****                                       */
/*  ***************       ***************                               */
/*  *****************   *****************                               */
/*  ***************       ***************                               */
/*          *****           *****           TheNetNode                  */
/*        *****               *****         Portable                    */
/*      *****                   *****       Network                     */
/*    *****                       *****     Software                    */
/*                                                                      */
/* File os/linux/kernelif.c (maintained by: DG9OBU)                     */
/*                                                                      */
/* This file is part of "TheNetNode" - Software Package                 */
/*                                                                      */
/* Copyright (C) 1998 - 2000 NORD><LINK e.V. Braunschweig               */
/*                                                                      */
/* This program is free software; you can redistribute it and/or modify */
/* it under the terms of the NORD><LINK ALAS (Allgemeine Lizenz fuer    */
/* Amateurfunk Software) as published by Hans Georg Giese (DF2AU)       */
/* on 13/Oct/1992; either version 1, or (at your option) any later      */
/* version.                                                             */
/*                                                                      */
/* This program is distributed WITHOUT ANY WARRANTY only for further    */
/* development and learning purposes. See the ALAS (Allgemeine Lizenz   */
/* fuer Amateurfunk Software).                                          */
/*                                                                      */
/* You should have received a copy of the NORD><LINK ALAS (Allgemeine   */
/* Lizenz fuer Amateurfunk Software) along with this program; if not,   */
/* write to NORD><LINK e.V., Hinter dem Berge 5, D-38108 Braunschweig   */
/*                                                                      */
/* Dieses Programm ist PUBLIC DOMAIN, mit den Einschraenkungen durch    */
/* die ALAS (Allgemeine Lizenz fuer Amateurfunk Software), entweder     */
/* Version 1, veroeffentlicht von Hans Georg Giese (DF2AU),             */
/* am 13.Oct.1992, oder (wenn gewuenscht) jede spaetere Version.        */
/*                                                                      */
/* Dieses Programm wird unter Haftungsausschluss vertrieben, aus-       */
/* schliesslich fuer Weiterentwicklungs- und Lehrzwecke. Naeheres       */
/* koennen Sie der ALAS (Allgemeine Lizenz fuer Amateurfunk Software)   */
/* entnehmen.                                                           */
/*                                                                      */
/* Sollte dieser Software keine ALAS (Allgemeine Lizenz fuer Amateur-   */
/* funk Software) beigelegen haben, wenden Sie sich bitte an            */
/* NORD><LINK e.V., Hinter dem Berge 5, D-38108 Braunschweig            */
/*                                                                      */
/************************************************************************/

#include "tnn.h"

#ifdef KERNELIF
#include "kernelif.h"

struct kip kip_info;

static void ifip_clearstat(void);
static void ifip_close(struct kip *);
static BOOLEAN ifip_setup(struct kip *);
static void ifip_init(struct kip *);
static BOOLEAN ifip_usable(struct kip *);
static BOOLEAN ifip_kernelip(struct kip *, char *);

/**************************************************************************/
/* Interface zur Steuerung der Kernelfunktionen                           */
/*------------------------------------------------------------------------*/
void ccpkif(void)
{
  MBHEAD *mbp;

  char hostname[32];

  bzero(&hostname[0], sizeof(hostname));

  /* Funktionen im Sysop-Modus */
  if (issyso())
  {
      if (strnicmp((char *)clipoi, "CLEAR", 5) == 0)
      {
          ifip_clearstat();

          mbp = putals("Interface statistics cleared.\r");
          prompt(mbp);
          seteom(mbp);
          return;
      }

      if (strnicmp((char *)clipoi, "DOWN", 4) == 0)
      {
          mbp = putals("Interface ");

          if (kip_info.if_active == TRUE)
          {
              ifip_close(&kip_info);

              if (xaccess("KIF_DOWN.TNB", 0) == 0)
                 runbatch("KIF_DOWN.TNB");

              rt_drop(kip_info.kernel_ip, 32);
              putstr("put to DOWN-state.\r", mbp);
          }
          else
              putstr("is DOWN.\r", mbp);

          prompt(mbp);
          seteom(mbp);
          return;
      }

      if (strnicmp((char *)clipoi, "UP", 2) == 0)
      {
          mbp = putals("Interface ");

          if (kip_info.if_available == FALSE)
          {
              putstr("is unavailable !!!\r", mbp);
              prompt(mbp);
              seteom(mbp);
              return;
          }

          if (kip_info.kernel_ip == 0L)
          {
              putstr("is not properly configured yet, set Kernel-IP with SETKIP !!!\r", mbp);
              prompt(mbp);
              seteom(mbp);
              return;
          }

          if (my_ip_addr == 0L)
          {
              putstr("is not properly configured yet, set Node-IP with IPA !!!\r", mbp);
              prompt(mbp);
              seteom(mbp);
              return;
          }

          if (kip_info.if_active == TRUE)
          {
              putstr("is already UP !\r", mbp);
              prompt(mbp);
              seteom(mbp);
              return;
          }

          if (ifip_setup(&kip_info) == FALSE)
              putstr("setup failed !!!\r", mbp);
          else
          {
              rt_add(kip_info.kernel_ip, 32, 0L, KERNEL_PORT, 0, 0, 0);

              if (xaccess("KIF_UP.TNB", 0) == 0)
                 runbatch("KIF_UP.TNB");

              putstr("setup successfully\r", mbp);
          }

          prompt(mbp);
          seteom(mbp);
          return;
      }

      if (strnicmp((char *)clipoi, "INIT", 4) == 0)
      {
          if (kip_info.if_active == TRUE)
          {
              mbp = putals("Interface is UP, put it to DOWN-state before initializing !!!\r");
              prompt(mbp);
              seteom(mbp);
              return;
          }

          mbp = putals("Kernel-Interface initilization ");

          ifip_init(&kip_info);

          if (ifip_usable(&kip_info) == FALSE)
              putstr("FAILED, feature not available !!!\r", mbp);
          else
              putstr("successfully\r", mbp);

          if (my_ip_addr == 0)
              putstr("WARNING: Node IP-Adress not yet set, set with IPA-Command !!!\r", mbp);

          prompt(mbp);
          seteom(mbp);
          return;
      }

      if (strnicmp((char *)clipoi, "STATUS", 5) == 0)
      {
          mbp = putals("Kernel-Interface status:\r");

          putstr("Kernel feature available : ", mbp);

          if (kip_info.if_available == FALSE)
              putstr("NO, not initialized or unavailable, try INIT.\r", mbp);
          else
          {
              putstr("yes, ",mbp);

              switch (kip_info.if_style)
              {
                  case 256: putstr("Kernel 2.4.x-style\r", mbp);
                            break;

                  default : putstr("Kernel 2.2.x-style\r", mbp);
              }
          }

          putstr("Kernel-Interface active  : ", mbp);

          if (kip_info.if_active == FALSE)
              putstr("no\r", mbp);
          else
              putstr("yes\r",mbp);

          putstr("IP-Adress of Kernel is   : ", mbp);

          if (kip_info.kernel_ip == 0)
              putstr("not defined\r", mbp);
          else
          {
              show_ip_addr(kip_info.kernel_ip, mbp);
              putchr('\r', mbp);
          }

          putstr("Bytes sent via Kernel    : ", mbp);
          putnum(kip_info.bytes_tx, mbp);

          putstr("\rBytes rcvd via Kernel    : ", mbp);
          putnum(kip_info.bytes_rx, mbp);

          putchr('\r', mbp);
          prompt(mbp);
          seteom(mbp);
          return;
      }

      if (strnicmp((char *)clipoi, "SETKIP", 6) == 0)
      {
          if (kip_info.if_active == TRUE)
          {
              mbp = putals("Interface is UP, put it DOWN before changing settings !!! \r");
              prompt(mbp);
              seteom(mbp);
              return;
          }

          nextspace(&clicnt, &clipoi);
          skipsp(&clicnt, &clipoi);

          sscanf(clipoi, "%s", hostname);

          mbp = putals("Kernel-IP ");

          if (ifip_kernelip(&kip_info, &hostname[0]) == FALSE)
              putstr("could not be set !!!\r", mbp);
          else
          {
              putstr("set to ", mbp);
              show_ip_addr(kip_info.kernel_ip, mbp);
              putchr('\r', mbp);
          }

          prompt(mbp);
          seteom(mbp);
          return;

      }

      mbp = putals("Usage: KERNELIF [INIT, CLEAR, SETKIP, UP, DOWN, STATUS]\r");
      prompt(mbp);
      seteom(mbp);
      return;
  }
  else
    invmsg();
}

/* Den Infoblock initialisieren */
static void ifip_init(struct kip *if_info)
{
    if_info->if_available = FALSE;
    if_info->if_style = 0;
    if_info->if_active = FALSE;
    if_info->kernel_ip = 0L;
    if_info->if_fd = -1;
    if_info->bytes_rx = 0L;
    if_info->bytes_tx = 0L;
}

/* Pruefen, ob notwendige Funktionen verfuegbar sind */
static BOOLEAN ifip_usable(struct kip *if_info)
{
    int fd = 0;
    int i = 0;

    char ifdev[14];

    if ((fd = open(IFIP, O_RDWR)) > 0)
    {
        if_info->if_available = TRUE;
        if_info->if_style = 256;
        close(fd);
        return TRUE;
    }

    for (i = 0; i < 255; i++)
    {
        sprintf(ifdev, "/dev/tun%d", i);
        if ((fd = open(ifdev, O_RDWR)) > 0)
        {
            if_info->if_available = TRUE;
            if_info->if_style = i;
            close(fd);
            return TRUE;
        }
    }

    if_info->if_available = FALSE;
    if_info->if_style = 0;
    return FALSE;
}

/* Den Link zum Kernel schliessen und im Infoblock als inaktiv vermerken */
static void ifip_close(struct kip *if_info)
{
    if ((if_info->if_active == TRUE) && (if_info->if_fd >= 0))
    {
        close(if_info->if_fd);
        if_info->if_fd = -1;
        if_info->if_active = FALSE;
    }
}

/* Das Interface kreieren */
BOOLEAN ifip_setup(struct kip *if_info)
{
    int fd = 0;
    int sd = 0;
    int err = 0;
    int i = 0;

    char dev[IFNAMSIZ];
    char ifdev[14];

    struct ifreq ifr;

    bzero(&ifr, sizeof(struct ifreq));

    if (if_info->if_style == 256)
    {
        sprintf(dev, "tnn");

        fd = open(IFIP, O_RDWR);

        ifr.ifr_flags = IFF_TUN | IFF_NO_PI;

        if (*dev)
           strncpy(ifr.ifr_name, dev, IFNAMSIZ);

        if ((err = ioctl(fd, TUNSETIFF, (void *) &ifr)) < 0 )
        {
           close(fd);
           return FALSE;
        }
    }
    else
    {
        sprintf(ifdev, "/dev/tun%d", if_info->if_style);
        fd = open(ifdev, O_RDWR);
        sprintf(dev, "tun%d", if_info->if_style);
    }

    if (fd <= 0)
        return FALSE;

    bzero(&ifr, sizeof(ifr));

    sd = socket(AF_INET, SOCK_DGRAM, 0);

    strcpy(ifr.ifr_name, dev);

    ifr.ifr_addr.sa_family = AF_INET;

    ifr.ifr_addr.sa_data[0] = 0;
    ifr.ifr_addr.sa_data[1] = 0;
    ifr.ifr_addr.sa_data[6] = 0;

    for (i = 0; i < 4; i++)
        ifr.ifr_addr.sa_data[5-i] = (if_info->kernel_ip >> 8 * i) & 0xff;

    if (ioctl(sd, SIOCSIFADDR, &ifr) < 0)
    {
        printf("kann kernel ip-adresse nicht setzen\n");
        close(sd);
        return FALSE;
    }

    for (i = 0; i < 4; i++)
        ifr.ifr_addr.sa_data[5-i] = (my_ip_addr >> 8 * i) & 0xff;

    if (ioctl(sd, SIOCSIFDSTADDR, &ifr) < 0)
    {
        printf("kann P-t-P ip-adresse nicht setzen\n");
        close(sd);
        return FALSE;
    }

    if ((err = ioctl(sd, SIOCGIFFLAGS, &ifr)) < 0)
    {
        printf("kann interface-flags nicht lesen\n");
        close(sd);
        return FALSE;
    }

    ifr.ifr_flags &= IFF_NOARP;
    ifr.ifr_flags |= IFF_UP;
    ifr.ifr_flags |= IFF_RUNNING;

    if ((err = ioctl(sd, SIOCSIFFLAGS, &ifr)) < 0)
    {
        printf("kann interface-flags nicht setzen\n");
        close(sd);
        return FALSE;
    }

    close(sd);

    strcpy(dev, ifr.ifr_name);

    if_info->if_active = TRUE;
    if_info->if_fd = fd;

    return TRUE;
}


/* Die Statistik des Infoblocks loeschen */
static void ifip_clearstat(void)
{
    kip_info.bytes_rx = 0L;
    kip_info.bytes_tx = 0L;
}

/* IP des Kernels im Infoblock eintragen */
static BOOLEAN ifip_kernelip(struct kip *if_info, char *hostname)
{
    int i = 0;
    struct hostent *hp = NULL;

    if ((hp = gethostbyname(hostname)) == NULL)
        return FALSE;

    if_info->kernel_ip = 0L;

    for (i = 0; i < 4; i++)
       if_info->kernel_ip = (if_info->kernel_ip << 8) | hp->h_addr_list[0][i];

    return TRUE;
}

/* Statistik ausgeben */
void ifip_dispstat(MBHEAD *mbp)
{
    putstr("\rKernel-Interface statistics:", mbp);

    putstr("\rBytes received : ", mbp);
    putnum(kip_info.bytes_rx, mbp);

    putstr("\rBytes sent     : ", mbp);
    putnum(kip_info.bytes_tx, mbp);

    putstr("\r", mbp);
}

/* Interface aktiv ? Wenn ja wird der verwendete Descriptor gemeldet, wenn */
/* nicht wird -1 zurueckgemeldet */
int ifip_active(void)
{
    if (kip_info.if_active == TRUE)
         return (kip_info.if_fd);
    else return (-1);
}

/* Ein IP-Frame vom Kernel holen und in den internen Router einschleusen. */
/* Sollte nur aufgerufen werden, wenn der Descriptor lesbar ist. */
void ifip_frame_to_router(void)
{
    char buf[2048];
    int i = -1;
    int j = 0;

    MBHEAD *mbhd;

    if ((i = read(kip_info.if_fd, &buf[0], 2048)) <= 0)
    {
        /* irgendwas lief schief, wir machen dicht */
        close(kip_info.if_fd);
        kip_info.if_fd = -1;
        kip_info.if_active = FALSE;

        return;
    }

    kip_info.bytes_rx += (ULONG)i;

    (mbhd = (MBHEAD *)allocb(ALLOC_MBHEAD))->l2port = KERNEL_PORT;

    for (j = 0; j < i; j++)
      putchr(buf[j], mbhd);

    rwndmb(mbhd);

    relink((LEHEAD *)mbhd, (LEHEAD *)iprxfl.tail);
}

void kif_frame_to_kernel(MBHEAD *mhbp)
{
    char buf[2048];
    int i = -1;
    int j = 0;

    /* falls das Interface nicht aktiv ist und noch Routing-Leichen da sind */
    /* die noch was auf dem Port abladen, einfach wegschmeissen */
    if (kip_info.if_active == FALSE)
    {
      dealmb(mhbp);
      return;
    }

    rwndmb(mhbp);

    while (mhbp->mbgc < mhbp->mbpc) /* solange Daten vorhanden sind */
        buf[j++] = getchr(mhbp);

    dealmb(mhbp);

    kip_info.bytes_tx += (ULONG)j;

    if ((i = write(kip_info.if_fd, &buf[0], j)) <= 0)
    {
        /* irgendwas lief schief, wir machen dicht */
        close(kip_info.if_fd);
        kip_info.if_fd = -1;
        kip_info.if_active = FALSE;
    }
}

#endif

/* End of os/linux/kernelif.c */
