/*
 * kissinit.c
 *
 * Program for setting up KISS mode on TNCs
 *
 * Copyright (C) 1997  Rene Stange, DG0FT, <stange@berlin.snafu.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*** Include ****************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <dos.h>
#include <conio.h>
#include <string.h>
#include <io.h>
#include <fcntl.h>

/*** Define *****************************************************************/

#define __VER__     "1.01 "

#define DEFPORT     "COM1"
#define DEFBAUD     9600
#define DEFFILE     "KISS.INI"

#define DELAYCHAR   30
#define DELAYLINE   300
#define DELAYEXIT   2000

#define PIC         0x20
#define PIT         0x40

#define DEFAULT     0xFFFF

/*** Macros *****************************************************************/

#define error(err, cond)    if (cond) errmsg (err)

#define BIOSCLK             *((wordfp) 0x46C)

#define COMBASE             ((wordfp) 0x400L)

#define WRCOM(reg, val)     outp (com_base + reg, val)
#define RDCOM(reg)          inp (com_base + reg)

/*** Types ******************************************************************/

typedef unsigned char       byte;
typedef unsigned short      word;
typedef unsigned long       dword;

typedef byte _far *         bytefp;
typedef word _far *         wordfp;

typedef enum {false, true}  bool;

/*** Globals ****************************************************************/

word com_port,
     com_base,
     com_baud   = DEFBAUD;

word delay_char = DELAYCHAR,
     delay_line = DELAYLINE,
     delay_exit = DELAYEXIT;

char *initfile  = DEFFILE;

bool verbose    = true,
     kiss_reset = false;

bool newline;

/*** Delay ******************************************************************/

dword getclk (void)
{
    word clkhigh, clklow, pending;

    _disable ();

    outp (PIC, 0xA);
    outp (PIT+3, 0);

    pending = inp (PIC) & 1;
    clklow  = inp (PIT);
    clklow  = ~(clklow | (inp (PIT) << 8));
    clkhigh = BIOSCLK;

    _enable ();

    if (pending && clklow <= 0xFF)
        clkhigh++;

    return (dword) clkhigh << 16 | clklow;
}

void delay (word msec)
{
    dword clk;

    clk = getclk () + msec * 1193L;

    while (getclk () < clk);
}

/*** COM Init ***************************************************************/

void com_init (void)
{
    word tconst;

    tconst = 115200L / com_baud;

    WRCOM (1, 0);
    WRCOM (4, 3);

    WRCOM (3, 0x83);
    WRCOM (0, tconst & 0xFF);
    WRCOM (1, tconst >> 8);
    WRCOM (3, 0x03);
}

/*** COM Reset **************************************************************/

void com_reset (void)
{
    WRCOM (4, 0);

    while (RDCOM (5) & 1)
        RDCOM (0);
}

/*** COM Write **************************************************************/

void com_write (word ch)
{
    if (ch < 256)
    {
        if (verbose)
        {
            newline = true;

            if (   ch >= ' '
                && ch <= '~'
                || ch == '\r')
            {
                putch (ch);

                if (ch == '\r')
                {
                    putch ('\n');

                    newline = false;
                }
            }
            else
                cprintf ("\\%u", ch);
        }

        while (!(RDCOM (5) & 0x20));

        WRCOM (0, ch);

        delay (ch == '\r' ? delay_line : delay_char);
    }
}

/*** Error Message **********************************************************/

enum {ERR_OPT,   ERR_PORT, ERR_DETECT, ERR_ADDR,   ERR_BAUD,
      ERR_DELAY, ERR_FILE, ERR_SECT,   ERR_LOADED, ERR_NUM};

char *msg[] = {"Invalid option (-H for help)",
               "Invalid port",
               "Port not present",
               "Port address out of range",
               "Invalid baud rate",
               "Delay out of range",
               "Can't open file",
               "Section not found",
               "Do not load TFPCX before",
               "Invalid numerical format"};

void errmsg (byte err)
{
    cprintf ("\nError: %s\r\n", msg[err]);
    exit (EXIT_FAILURE);
}

/*** Signon *****************************************************************/

void signon (void)
{
    cprintf ("Ŀ\r\n"
             " KISSINIT v"__VER__"("__DATE__") by DG0FT \r\n"
             "Ĵ\r\n"
             "  This is free software without any    \r\n"
             "  warranty. You can redistribute it    \r\n"
             "  under the terms of the GNU General   \r\n"
             "  Public License (see LICENSE.DOC).    \r\n"
             "Ĵ\r\n"
             " COM%u (%X), %5u Bd, %3u %3u %4u ms \r\n"
             "\r\n"

             "\n%s ...\r\n\n",

             com_port, com_base, com_baud,
             delay_char, delay_line, delay_exit,
             kiss_reset ? "KISS reset" : "Reading");
}

/*** Help *******************************************************************/

char help[] =

"\n"
"Options:                                  Default\r\n"
"\n"
"  -PCOMn[:xxx]    COM port [address]      COM1\r\n"
"  -Bnnnn          baud rate               9600\r\n"
"  -Dnnn:nnn:nnnn  delay (char/line/exit)  30:300:2000 (ms)\r\n"
"  -F<file>        init file               KISS.INI\r\n"
"  -S<section>     choose section in file\r\n"
"  -K              kiss reset\r\n"
"  -N              no messages\r\n";

/*** Token ******************************************************************/

char *token (char *str)
{
    static char *pos;

    if (str)
        pos = str;
    else
        str = pos;

    while (*pos && *pos != ':')
        pos++;

    if (*pos)
        *pos++ = '\0';

    return str;
}

/*** Number *****************************************************************/

word number (char *str, byte base)
{
    dword value = 0;
    byte digit;

    if (*(str = token (str)))
    {
        while (*str)
        {
            if ((digit = *str++ - '0') > 9)
                error (ERR_NUM, (digit -= 7) < 10 || digit >= base);

            error (ERR_NUM, (value = value * base + digit) >= DEFAULT);
        }

        return (word) value;
    }

    return DEFAULT;
}

/*** COM Config *************************************************************/

void com_cfg (char *arg)
{
    error (ERR_PORT,    strncmp (arg, "COM", 3)
                     || (com_port = number (arg+3, 10))-1 > 3
                     || arg[4]);

    if ((com_base = number (NULL, 16)) != DEFAULT)
    {
        error (ERR_ADDR,    com_base < 0x100
                         || com_base > 0x3F8
                         || com_base & 7);
    }
    else
        error (ERR_DETECT, !(com_base = COMBASE[com_port-1]));
}

/*** Delay Config ***********************************************************/

void delay_cfg (char *arg, word *var, word max)
{
    word delay;

    if ((delay = number (arg, 10)) != DEFAULT)
    {
        error (ERR_DELAY, delay > max);

        *var = delay;
    }
}

/*** TFPCX loaded? **********************************************************/

byte hdr_tfpc[] = "???N5NX",
     hdr_drsi[] = "\x53\x1E\xBB??\x8E\xDB\x84\xE4\x74\x20";

bool loaded (void)
{
    byte vector, hdr, *p1;
    bytefp p2, handler;

    for (vector = 0xFF; vector >= 0x40; vector--)
    {
        handler = (bytefp) _dos_getvect (vector);

        for (hdr = 0, p1 = hdr_tfpc; hdr < 2; hdr++, p1 = hdr_drsi)
            for (p2 = handler; *p1 == '?' || *p1 == *p2; p2++)
                if (!*++p1)
                    return true;
    }

    return false;
}

/*** KISS off ***************************************************************/

void kiss_off (void)
{
    com_write (0xC0);
    com_write (0xFF);
    com_write (0xC0);
}

/*** Upcase *****************************************************************/

byte upcase (byte ch)
{
    return 'a' > ch || 'z' < ch ? ch : ch - ('a'-'A');
}

/*** Read Init **************************************************************/

void read_init (word handle)
{
    byte ch, digit, count, state = 0;
    word buf;

    while (state != 6 && read (handle, &ch, 1) == 1)
    {
        if (ch == '\n')
            ch = '\r';

        switch (state)
        {
        case 0:
            switch (ch)
            {
            case ';':
                state = 5;
                continue;

            case '[':
                state = 6;
                continue;

            default:
                state = 1;
            }

        case 1:
        case_1:
            switch (ch)
            {
            case '\\':
                state = 2;
                break;

            case '^':
                state = 4;
                break;

            default:
                com_write (ch);
            }
            break;

        case 2:
            state = 1;

            switch (upcase (ch))
            {
            case '\r':
                break;

            case 'D':
                delay (1000);
                break;

            case 'E':
                com_write (0x1B);
                break;

            case 'K':
                kiss_off ();
                break;

            default:
                if ((buf = ch - '0') < 10)
                {
                    count = 2;
                    state = 3;
                }
                else
                    com_write (ch);
            }
            break;

        case 3:
            if ((digit = ch - '0') < 10)
            {
                buf = buf * 10 + digit;

                if (--count)
                    break;
            }

            com_write (buf);

            state = 1;

            if (digit > 9)
                goto case_1;

            break;

        case 4:
            if ((buf = upcase (ch) - '@') < ' ')
                com_write (buf);

            state = 1;
        }

        if (ch == '\r')
            state = 0;
    }

    if (state == 3)
        com_write (buf);

    close (handle);
}

/*** Search *****************************************************************/

bool search (word handle, byte *section)
{
    byte ch, *sect, state = 0;

    while (read (handle, &ch, 1) == 1)
    {
        if (ch == '\n')
            ch = '\r';

        switch (state)
        {
        case 0:
            if (ch == '[')
            {
                sect = section;
                state = 1;
            }
            else
                state = 3;

            break;

        case 1:
            if (*sect)
            {
                if (upcase (ch) != *sect++)
                    state = 3;
            }
            else
                if (ch == ']')
                    state = 2;
                else
                    state = 3;

            break;

        case 2:
            if (ch == '\r')
                return true;
        }

        if (ch == '\r')
            state = 0;
    }

    close (handle);

    return false;
}

/*** Main *******************************************************************/

void main (word argc, char **argv)
{
    byte *port = DEFPORT, *section = NULL;
    word handle;

    while (*++argv)
    {
        error (ERR_OPT, *strupr((*argv)++) != '-');

        switch (*(*argv)++)
        {
        case 'B':
            com_baud = number (*argv, 10);

            error (ERR_BAUD,    com_baud !=  2400 && com_baud !=  4800
                             && com_baud !=  9600 && com_baud != 19200
                             && com_baud != 38400 && com_baud != 57600);
            break;

        case 'D':
            delay_cfg (*argv, &delay_char, 999);
            delay_cfg (NULL,  &delay_line, 999);
            delay_cfg (NULL,  &delay_exit, 9999);
            break;

        case 'F':
            initfile = *argv;
            break;

        case 'H':
            cputs (help);
            exit (EXIT_SUCCESS);

        case 'K':
            kiss_reset = true;
            break;

        case 'N':
            verbose = false;
            break;

        case 'P':
            port = *argv;
            break;

        case 'S':
            section = *argv;
            break;

        default:
            errmsg (ERR_OPT);
        }
    }

    com_cfg (port);

    error (ERR_LOADED, loaded ());

    if (!kiss_reset)
    {
        error (ERR_FILE, (handle = open (initfile, O_RDONLY | O_TEXT)) == -1);

        error (ERR_SECT, section && !search (handle, section));
    }

    if (verbose)
        signon ();

    com_init ();

    if (kiss_reset)
        kiss_off ();
    else
        read_init (handle);

    delay (delay_exit);

    com_reset ();

    if (newline)
        cputs ("\r\n");

    exit (EXIT_SUCCESS);
}

/****************************************************************************/
