/*
 * tfpcx.c
 *
 * The Firmware PC Extended (TFPCX)
 *
 * Copyright (C) 1997  Rene Stange, DG0FT, <stange@berlin.snafu.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*** Include ****************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <dos.h>
#include <conio.h>
#include <string.h>
#include <malloc.h>
#include <io.h>
#include <fcntl.h>

/*** Define *****************************************************************/

#define __VER__     "2.21 "     /* version string */
#define VERSION     0x215       /* version number */

#define L1PORTS     8           /* number of L1 ports */
#define LINKNMBR    40          /* max. number of channels */
#define DEFLINKS    10          /* default number of channels */
#define MB_SIZE     36          /* size of message buffer */
#define LB_SIZE     112         /* size of link block */
#define TF_BUFS     600         /* number of buffers */
#define HOSTBUF     512         /* must be a power of 2 */
#define STACK       256         /* words */

/* Ports */
#define PIC0        0x20        /* Master PIC */
#define PIC1        0xA0        /* Slave PIC */
#define PIT         0x40        /* Timer Port 0 */
#define PPI         0x61        /* Parallel Port (Beeper) */

#define EOI         0x20        /* end of interrupt */
#define SETPRIO     0xC0        /* set IRQ priority */

#define IRQ0        0x08        /* int vector of IRQ0 */
#define IRQ8        0x70        /* int vector of IRQ8 */

/* Modem */
#define MODEMS      2

/* PAR */
#define TXD         0x01
#define PTT         0x02
#define BURST       0x04
#define DCDBIT      0x10
#define RXD         0x20

#define IRQEN       0x10

#define DCDCOUNT    5
#define DCDABORT    5
#define DCDLIMIT    15

/* SCC */
#define SCCCHANS    4

#define PCLK        4915200L
#define TC75HZ      65534       /* 75 Hz zero count int */

/* KISS */
#define KISSPORTS   4

#define FEND        0xC0
#define FESC        0xDB
#define TFEND       0xDC
#define TFESC       0xDD

#define DATA        0x00
#define RMNC        0x20
#define SMACK       0x80

#define TPAR        1
#define PPAR        2
#define WPAR        3
#define TAPAR       4
#define DPAR        5

/* L1 */
#define MIN_LEN     8

#define STUFF       5
#define FLAG        6
#define ABRT        7
#define FREE        30

/* CRC */
#define CRC_RESET   0xFFFF
#define CRC_MASK    0x8408
#define CRC_CHECK   0xF0B8

/*** Z8530 ******************************************************************/

/* Write Registers */

#define RES_EX_INT  0x10        /* WR0 */
#define RES_TX_IP   0x28
#define RES_ERR     0x30
#define RES_TX_CRC  0x80
#define RES_EOM     0xC0

#define EX_IE       0x01        /* WR1 */
#define TX_IE       0x02
#define RX_IE       0x10

#define RX_DIS      0xC8        /* WR3 */
#define RX_EN       0xC9

#define SDLC        0x20        /* WR4 */

#define TX_DIS      0x61        /* WR5 */
#define TX_EN       0x69
#define RTS         0x02

#define SDLC_FLAG   0x7E        /* WR7 */

#define MINT_EN     0x0A        /* WR9 */
#define HW_RES      0xC0

#define NRZ         0x80        /* WR10 */
#define NRZI        0xA0
#define ABUNDER     0x04

#define TRxC_RTxC   0x20        /* WR11 */
#define DPLL_RTxC   0x64
#define DPLL_BRG    0x74
#define TRxOUT_BRG  0x02
#define TRxOUT_DPLL 0x03

#define DPLL_EN     0x23        /* WR14 */
#define DPLL_DIS    0x60
#define DPLL_SRCBRG 0x80
#define DPLL_NRZI   0xE0
#define BRG_EN      0x63

#define ZC_IE       0x02        /* WR15 */
#define DCD_IE      0x08
#define SYNC_IE     0x10
#define CTS_IE      0x20


/* Read Registers */

#define DCD         0x08        /* RR0 */
#define HUNT        0x10
#define CTS         0x20
#define TX_EOM      0x40
#define ABORT       0x80

#define RESIDUE     0x0E        /* RR1 */
#define RES8        0x06
#define RX_OVR      0x20
#define CRC_ERR     0x40
#define SDLC_EOF    0x80

/*** Macros *****************************************************************/

#define LOBYTE(val)         ((val) & 0xFF)
#define HIBYTE(val)         ((val) >> 8)

#define intr                _cdecl _interrupt _far

#define idisable()          { _asm  pushf \
                              _asm  cli  }
#define irestore()          { _asm  popf }

#define debug(major)        if (debugdev == major) \
                                outp (PPI, inp (PPI) & 0xFE ^ 2)

#define tfpcx_avl()         (drsi ? HIBYTE (chbuf = call_tfpcx (0x000)) \
                                  : LOBYTE (        call_tfpcx (0x100)))
#define tfpcx_inp()         (LOBYTE (drsi ? chbuf : call_tfpcx (0x200)))
#define tfpcx_out(val)      call_tfpcx ((drsi ? 0x100 : 0x300) | val)

#define error(err, cond)    if (cond) errmsg (err)

#define set(var, bit)       var |=  (1 << bit)
#define res(var, bit)       var &= ~(1 << bit)

#define VIDEO               (*(bytefp) 0x449 == 7 \
                            ? (bytefp) 0xB000009F \
                            : (bytefp) 0xB800009F)

/* IRQ control */
#define irqvec(irq)         (irq + (irq < 8 ? IRQ0 : IRQ8-8))

#define maskreg(irq)        (irq < 8 ? PIC0+1 : PIC1+1)

#define irqmask(irq)        (1 << (irq & 7))

#define eoi(irq)            if (irq >= 8) \
                                outp (PIC1, EOI); \
                            outp (PIC0, EOI)

#define irq_prio(irq)       outp (PIC0, SETPRIO | (irq-1) & 7)

#define isat()              (*(bytefp) 0xF000FFFE == 0xFC)

/* Modem */
#define BIOSBASE            ((wordfp) (modem->rs232 ? 0x400L : 0x408L))

/* PAR */
#define LPTBASE             ((wordfp) 0x408L)

/* RTC */
#define WRRTC(reg, val)     outp ((outp (0x70, reg), 0x71), val)
#define RDRTC(reg)          inp ((outp (0x70, reg), 0x71))

/* SCC */
#define SCCDELAY            inp (0xE4)      /* non-existing port (2s) */

#define WR0(val)            outp ((SCCDELAY, scc->ctrl), val)
#define WR8(val)            WR (8, val)
#define WR(reg, val)        outp ((SCCDELAY, outp (scc->ctrl, reg), \
                                   SCCDELAY, scc->ctrl), val)

#define RR0                 inp ((SCCDELAY, scc->ctrl))
#define RR8                 inp ((SCCDELAY, scc->data))
#define RR(reg)             inp ((SCCDELAY, outp (scc->ctrl, reg), \
                                  SCCDELAY, scc->ctrl))

#define not0(par)           (par ? par : 1)

/* KISS */
#define COMBASE             ((wordfp) 0x400L)

#define WRCOM(reg, val)     outp (kiss->base + reg, val)
#define RDCOM(reg)          inp (kiss->base + reg)

/* L2 */
#define rxput(port, val)    l1put ((port) << 8 | (val))
#define rxeof(port, ignore) l1put ((port) << 8 | 0x8000 | (ignore) << 1)
#define rxdiscard(port)     l1put ((port) << 8 | 0x8001)

#define txget(port)         l1get ((port) << 8)
#define txrewind(port)      l1get ((port) << 8 | 0x8000)
#define txdiscard(port)     l1get ((port) << 8 | 0x8001)

#define TXEOF               0x8000
#define TXLAST              0x8001

/*** Enum *******************************************************************/

enum {LOOP, MODEM, PAR, SCC, KISS, RTC, NODEV};             /* Major Ports */

enum {L1NAME, L1TYPE, L1ADDR, L1IRQ, L1BAUD, L1ERR, L1CLR}; /* L1 Infos */

enum {DSCC, OSCC, USCC};                                    /* SCC Types */

enum {DISABLE, SOFTCLK, HARDCLK, DF9IC, PA0HZP, CLKCHAN};   /* Modem Types */

enum {IDLE, DWAIT, DELAY, ACTIVE, FLUSH, TAIL};             /* TX States */

enum {CRCOFF, CRCSMACK, CRCRMNC};                           /* CRC Modes */

/*** Types ******************************************************************/

typedef unsigned char       byte;
typedef unsigned short      word;
typedef unsigned long       dword;

typedef byte _far *         bytefp;
typedef word _far *         wordfp;

typedef enum {false, true}  bool;

typedef void (intr *procfp) ();

/*** Struct *****************************************************************/

typedef struct buf_t
{
    struct buf_t *next;
    byte data[];
}
buf_t;

typedef struct modem_t
{
    byte l2port;

    byte port;
    bool rs232;
    word port_out, port_inp;
    byte ptt_on, ptt_off, data_out, data_inp;

    word baud, baud_delay, baud_cnt, send_cnt;

    byte dcd_cnt, bit_cnt, idle, shift, dpll, level;
    word frm_len, crc, fcs;
    bool dcd, sync;

    byte tx_state, tx_bitcnt, tx_idle;
    word tx_timer, tx_fcs, tx_shift;
    bool tx_kicked;
}
modem_t;

typedef struct par_t
{
    byte l2port;

    byte port;
    word port_out, port_inp;
    byte irq;
    procfp old_int;

    word baud;

    byte bit_cnt, idle, shift, lastbit;
    word frm_len, crc, fcs, dcd_cnt, dcd_timer;
    bool dcd, sync;

    byte tx_state, tx_bitcnt, tx_idle, tx_bit;
    word tx_timer, tx_fcs, tx_shift;
    bool tx_kicked;

    dword scr;
}
par_t;

typedef struct scc_t
{
    byte l2port;

    byte chan;
    word data, ctrl;
    bool clkchan;

    word baud;
    byte modem;

    bool dcd, rx_err;
    word rx_len, dcd_timer;

    byte tx_state;
    word tx_timer;

    byte rr0;

    word errors;
}
scc_t;

typedef struct sccboard_t
{
    word base;
    byte irq;
    char modems[5];

    byte data[4];
    short ctrl;
}
sccboard_t;

typedef struct kiss_t
{
    byte l2port;

    byte port;
    word base;
    byte irq;
    procfp old_int;

    word baud;

    byte rx_state, tx_state;
    word rx_len, dcd_timer;
    bool tx_kicked;
    byte fesc;

    bool crc_ok;
    byte tx_mode, rx_mode;
    word tx_crc, rx_crc;

    word errors;

    buf_t *parlst;
    byte bufp;
}
kiss_t;

typedef struct port_t
{
    byte major;

    union
    {
        void    *minor;
        modem_t *modem;
        par_t   *par;
        scc_t   *scc;
        kiss_t  *kiss;
    };
}
port_t;

/*** Prototypes *************************************************************/

void l2 (void);                             /* TF */
void l3 (void);
void lx (void);
void mainf (void);
bool l1put (word);
word l1get (word);
bool info_avl (void);
buf_t *allocb (void);
void dealoc (buf_t *);

void intr iface_tfpc (void);                /* IFACE */
void intr iface_drsi (void);
word call_tfpcx (word);
void   _cdecl chain (procfp);
byte   _cdecl random (void);
wordfp _cdecl set_stk (wordfp);
bool   _cdecl mtasker (void);

void int_tx (scc_t *);                      /* SCC */
void int_ex (scc_t *);
void int_rx (scc_t *);
void int_sp (scc_t *);

/*** TF Parameters **********************************************************/

byte defESC   = 0x1B,
     defIp[7] = "      \x60",
     defYp    = DEFLINKS,
     defA1p   = 7,
     defA2p   = 15,
     defA3p   = 3,
     defIPp   = 80,
     defMp    = 0,
     defRp    = 1,
     defPp    = 32,
     defWp    = 10,
     defTp    = 25,
     defXp    = 1,
     defVp    = 1,
     defOp    = 2,
     defNp    = 20,
     defUIp   = 1,                          /* UI-Poll */
     defVCp   = 0,
     defCp    = 0,
     defDp    = 0,
     defEp    = 1,
     defM7p   = 0,
     defxSp   = 0;

word defFp    = 250,
     defDAp   = 120,
     defT2p   = 150,
     defT2d   = 1,                          /* T2 if DAMA enabled */
     defT3p   = 30000,
     defTAp   = 1;                          /* TX Tail */

/*** Default Parameters *****************************************************/

word defInt   = 0xFD,                       /* TFPCX interrupt */
     bufcnt   = TF_BUFS,
     linknmbr = DEFLINKS;

bool logo     = true,                       /* Options */
     blink    = true,
     loopback = true,
     carrier  = false,
     drsi     = false,
     drmsg    = false,
     dxmsg    = false,
     stampok  = false;

byte lastdev  = LOOP,                       /* Last attached device */
     debugdev = NODEV,                      /* Debug device */

     clockdev = LOOP,                       /* Clock device */
     clkprio  = 18,                         /* Ticks per second */
     clkstep  = 6;                          /* max. increment of ticks */

/* Ports */
byte l1ports;
port_t l1port[L1PORTS];

byte modems;
modem_t modemtab[MODEMS], *modemend = modemtab;
word pit0_const, int8_delay, tick_delay;

byte parports;
par_t partab;

byte sccs, sccirq, scctype = DSCC;
scc_t scctab[SCCCHANS], *sccend = scctab, *sccclkchan = &scctab[0];

byte kissports;
kiss_t kisstab[KISSPORTS], *kissend = kisstab;

/*** Globals ****************************************************************/

extern byte Cpar[], Dpar[], Ppar[], Tpar[], Wpar[], Xpar[], ishmod;
extern word TApar[], ticks;
extern void *lnktbl;

procfp old_int8, old_int13, old_int1C, old_int21,
       old_rtcint, old_sccint, old_defInt;

byte *bufbgn, *bufend;

byte hibuf[HOSTBUF], hobuf[HOSTBUF];
word hihead, hitail, hohead, hotail;

word local_stack[STACK];
wordfp global_stack;
bool semaphore;

byte rxinfo, txinfo, attrib;
bool l2call;
word chbuf;

/*** IRQ Init ***************************************************************/

procfp irq_init (byte irq, procfp handler)
{
    procfp old_int;
    byte vec, pic;

    old_int = _dos_getvect (vec = irqvec (irq));
    _dos_setvect (vec, handler);
    outp (pic, inp (pic = maskreg (irq)) & ~irqmask (irq));

    return old_int;
}

/*** IRQ Reset **************************************************************/

void irq_reset (byte irq, procfp handler)
{
    byte pic;

    outp (pic, inp (pic = maskreg (irq)) | irqmask (irq));
    _dos_setvect (irqvec (irq), handler);
}

/*** Level 2 ****************************************************************/

void level_2 (void)
{
    _disable ();

    if (!semaphore)
    {
        semaphore = true;
        _enable ();
        global_stack = set_stk (local_stack + STACK);

        l2 ();
        l3 ();

        set_stk (global_stack);
        semaphore = false;
    }
    else
        _enable ();
}

/*** Level X ****************************************************************/

void _cdecl level_x (void)
{
    _disable ();

    if (!semaphore)
    {
        semaphore = true;
        _enable ();
        global_stack = set_stk (local_stack + STACK);

        lx ();

        set_stk (global_stack);
        semaphore = false;
    }
    else
        _enable ();
}

/*** L1 Info ****************************************************************/

word l1info (byte port, byte info)
{
    static char *modems[] = {"SOFTCLK", "HARDCLK", "DF9IC", "PA0HZP"};
    modem_t *modem;
    kiss_t *kiss;
    scc_t *scc;
    char *name;

    switch (l1port[port].major)
    {
    case MODEM:
        modem = l1port[port].modem;

        switch (info)
        {
        case L1NAME: name    = modem->rs232 ? "COM1" : "LPT1";
                     name[3] = modem->port + '1';
                     return (word) name;

        case L1TYPE: return (word) "MODEM";
        case L1ADDR: return modem->port_out - (modem->rs232 ? 4 : 0);
        case L1BAUD: return modem->baud;
        }
        break;

    case PAR:
        switch (info)
        {
        case L1NAME: name    = "LPT1";
                     name[3] = partab.port + '1';
                     return (word) name;

        case L1TYPE: return (word) "PAR96";
        case L1ADDR: return partab.port_out;
        case L1IRQ:  return partab.irq;
        case L1BAUD: return partab.baud;
        }
        break;

    case SCC:
        scc = l1port[port].scc;

        switch (info)
        {
        case L1NAME: name    = "SCC0";
                     name[3] = scc->chan + '0';
                     return (word) name;

        case L1TYPE: return (word) modems[scc->modem - 1];
        case L1ADDR: return scctype == OSCC ? scc->ctrl : scc->data;
        case L1IRQ:  return sccirq;
        case L1BAUD: return scc->baud;

        case L1CLR:  scc->errors = 0;
        case L1ERR:  return scc->errors;
        }
        break;

    case KISS:
        kiss = l1port[port].kiss;

        switch (info)
        {
        case L1NAME: name    = "COM1+";
                     name[3] = kiss->port + '1';
                     name[4] = kiss->crc_ok ? '+' : '\0';
                     return (word) name;

        case L1TYPE: return (word) "KISS";
        case L1ADDR: return kiss->base;
        case L1IRQ:  return kiss->irq;
        case L1BAUD: return kiss->baud;

        case L1CLR:  kiss->errors = 0;
        case L1ERR:  return kiss->errors;
        }
    }

    return info == L1NAME ? (word) "NULL" : 0;
}

/*** L1 Parameter ***********************************************************/

void l1par (byte port, byte par, word val)
{
    buf_t *parbuf, *lastbuf;
    kiss_t *kiss;

    if (l1port[port].major == KISS)
    {
        kiss = l1port[port].kiss;

        parbuf = allocb ();
        parbuf->next = NULL;

        parbuf->data[0] = FEND;
        parbuf->data[1] = par;
        parbuf->data[2] = val <= 255 ? val : 255;
        parbuf->data[3] = FEND;

        idisable ();

        if (lastbuf = kiss->parlst)
        {
            while (lastbuf->next)
                lastbuf = lastbuf->next;

            lastbuf->next = parbuf;
        }
        else
        {
            kiss->parlst = parbuf;

            if (!kiss->tx_state)
            {
                WRCOM (0, parbuf->data[kiss->bufp++]);
                set (txinfo, kiss->l2port);
                kiss->tx_state = 6;
            }
        }

        irestore ();
    }
}

/*** Is DCD? ****************************************************************/

bool iscd (byte port)
{
    if (!Dpar[port])
    {
        switch (l1port[port].major)
        {
        case MODEM: return l1port[port].modem->dcd;
        case PAR:   return partab.dcd;
        case SCC:   return l1port[port].scc->dcd;
        }
    }

    return false;
}

/*** Kick TX ****************************************************************/

void kicktx (byte port)
{
    kiss_t *kiss;
    scc_t *scc;

    if (Xpar[port])
    {
        switch (l1port[port].major)
        {
        case MODEM:
            l1port[port].modem->tx_kicked = true;
            return;

        case PAR:
            idisable ();

            partab.tx_kicked = true;

            if (!partab.tx_state)
            {
                partab.tx_timer = Dpar[port] ? 1 : not0 (Wpar[port]);
                partab.tx_state = DWAIT;
            }

            irestore ();
            return;

        case SCC:
            scc = l1port[port].scc;

            idisable ();

            switch (scc->tx_state)
            {
            case IDLE:
                scc->tx_timer = Dpar[port] ? 1 : not0 (Wpar[port]);
                scc->tx_state = DWAIT;
                break;

            case TAIL:
                scc->tx_timer = 1;
                scc->tx_state = DELAY;
            }

            irestore ();
            return;

        case KISS:
            kiss = l1port[port].kiss;

            idisable ();

            if (kiss->tx_state)
                kiss->tx_kicked = true;
            else
            {
                WRCOM (0, FEND);
                set (txinfo, kiss->l2port);
                kiss->tx_state = 1;
            }

            irestore ();
            return;
        }
    }

    txdiscard (port);
}

/*** SCC Clock **************************************************************/

void scc_clk (scc_t *scc, bool tx_on)
{
    word tconst;

    if (scc->modem == DF9IC)
    {
        WR (11, TRxC_RTxC);
        WR (14, DPLL_DIS);
    }
    else
    {
        tconst = PCLK / 64L / scc->baud;
        if (tx_on)
            tconst *= 32;
        tconst -= 2;

        WR (12, LOBYTE (tconst));
        WR (13, HIBYTE (tconst));

        WR (11, tx_on ? DPLL_BRG | TRxOUT_BRG
                      : (scc->modem == PA0HZP ? DPLL_RTxC | TRxOUT_BRG
                                              : DPLL_RTxC | TRxOUT_DPLL));

        WR (14, DPLL_SRCBRG);
        WR (14, DPLL_NRZI);
        WR (14, DPLL_EN);
    }
}

/*** SCC TX *****************************************************************/

void scc_tx (scc_t *scc, bool tx_on)
{
    if (scc->modem == SOFTCLK)
    {
        WR (3, RX_DIS);
        WR (5, TX_DIS);

        scc_clk (scc, tx_on);

        if (tx_on)
            WR (5, TX_EN | RTS);
        else
            WR (3, RX_EN);
    }
    else
        WR (5, tx_on ? TX_EN | RTS : TX_EN);
}

/*** Timer ******************************************************************/

void timer (word ticks)
{
    static word serv_cnt = 10, blnk_cnt = 20;
    bytefp video = VIDEO;
    modem_t *modem;
    kiss_t *kiss;
    scc_t *scc;
    byte port;

    for (modem = modemtab; modem < modemend; modem++)
    {
        _disable ();

        if (modem->tx_timer)
            if (modem->tx_timer <= ticks)
                modem->tx_timer = 0;
            else
                modem->tx_timer -= ticks;

        _enable ();
    }

    if (parports)
    {
        _disable ();

        if (partab.tx_timer)
            if (partab.tx_timer <= ticks)
            {
                partab.tx_timer = 0;

                if (partab.tx_state == DWAIT)
                    if (   Dpar[partab.l2port]
                        ||    !partab.dcd
                           && random () <= Ppar[partab.l2port])
                    {
                        set (txinfo, partab.l2port);
                        partab.tx_timer = Tpar[partab.l2port];
                        partab.tx_state = DELAY;
                    }
                    else
                        partab.tx_timer = not0 (Wpar[partab.l2port]);
            }
            else
                partab.tx_timer -= ticks;

        if (partab.dcd_timer)
            if (partab.dcd_timer <= ticks)
            {
                partab.dcd_timer = 0;

                set (rxinfo, partab.l2port);
            }
            else
                partab.dcd_timer -= ticks;

        _enable ();
    }

    for (scc = scctab; scc < sccend; scc++)
        if (scc->modem)
        {
            _disable ();

            if (scc->tx_timer)
                if (scc->tx_timer <= ticks)
                {
                    scc->tx_timer = 0;

                    switch (scc->tx_state)
                    {
                    case DWAIT:
                        if (Dpar[scc->l2port] || !scc->dcd
                            && random () <= Ppar[scc->l2port])
                        {
                            scc_tx (scc, true);
                            set (txinfo, scc->l2port);
                            scc->tx_timer = not0 (Tpar[scc->l2port]);
                            scc->tx_state = DELAY;
                        }
                        else
                            scc->tx_timer = not0 (Wpar[scc->l2port]);
                        break;

                    case DELAY:
                        WR0 (RES_TX_CRC);
                        WR8 (txget (scc->l2port));
                        WR0 (RES_EOM);
                        scc->tx_state = ACTIVE;
                        break;

                    case TAIL:
                        scc_tx (scc, false);
                        res (txinfo, scc->l2port);
                        scc->tx_state = IDLE;
                    }
                }
                else
                    scc->tx_timer -= ticks;

            if (scc->dcd_timer)
                if (scc->dcd_timer <= ticks)
                {
                    scc->dcd_timer = 0;

                    set (rxinfo, scc->l2port);
                }
                else
                    scc->dcd_timer -= ticks;

            _enable ();
        }

    for (kiss = kisstab; kiss < kissend; kiss++)
    {
        _disable ();

        if (kiss->dcd_timer)
            if (kiss->dcd_timer <= ticks)
            {
                kiss->dcd_timer = 0;

                res (rxinfo, kiss->l2port);
            }
            else
                kiss->dcd_timer -= ticks;

        _enable ();
    }

    if (serv_cnt <= ticks || l2call)
    {
        serv_cnt = 10;              /* 10/s */
        l2call = false;
        level_2 ();
    }
    else
        serv_cnt -= ticks;

    if (ishmod)
    {
        if (carrier && l1ports)
            for (port = 1 << l1ports-1; port; port >>= 1, video--)
            {
                if (attrib)
                    *video = attrib;
                *--video = txinfo & port ? 'S' : (rxinfo & port ? 'R' : '');
            }
    }
    else
    {
        if (blnk_cnt <= ticks)
        {
            blnk_cnt = 20;          /* 5/s */

            if (info_avl () && blink)
                *video ^= 0x77;
        }
        else
            blnk_cnt -= ticks;
    }
}

/*** SCC Interrupt **********************************************************/

void (*handler[]) (scc_t *) = {int_tx, int_ex, int_rx, int_sp};

void intr scc_int (void)
{
    static word oldticks;
    word newticks;
    scc_t *scc;
    byte rr2;

    debug (SCC);

    _disable ();
    eoi (sccirq);

    for (scc = scctab; scc < sccend;)
        if (RR (3))
        {
            scc++;
            rr2 = RR (2);
            handler[rr2 >> 1 & 3] (&scctab[rr2 >> 3 ^ 1]);
            scc = scctab;
        }
        else
            scc += 2;

    if (clockdev == SCC && (newticks = ticks - oldticks))
    {
        oldticks = ticks;
        _enable ();
        timer (newticks);
    }
}

/*** TX Handler *************************************************************/

void int_tx (scc_t *scc)
{
    word val;

    switch (scc->tx_state)
    {
    case ACTIVE:
        if (RR0 & TX_EOM)
        {
            scc->errors++;

            txrewind (scc->l2port);
            scc->tx_state = FLUSH;
        }
        else
            if ((val = txget (scc->l2port)) & TXEOF)
            {
                WR (10, scc->modem == DF9IC ? NRZ : NRZI);
                WR0 (RES_TX_IP);
                scc->tx_state = FLUSH;
            }
            else
                WR8 (val);
        break;

    case FLUSH:
        WR (10, scc->modem == DF9IC ? NRZ | ABUNDER : NRZI | ABUNDER);

        if ((val = txget (scc->l2port)) < TXEOF)
        {
            WR0 (RES_TX_CRC);
            WR8 (val);
            WR0 (RES_EOM);
            scc->tx_state = ACTIVE;
        }
        else
        {
            WR0 (RES_TX_IP);
            scc->tx_timer = TApar[scc->l2port] + clkstep;
            scc->tx_state = TAIL;
        }
    }
}

/*** Extern Status Change ***************************************************/

void int_ex (scc_t *scc)
{
    static byte cnt;
    byte rr0, delta;

    delta = (rr0 = RR0) ^ scc->rr0;

    if (scc->modem)
    {
        if (scc->rx_len && rr0 & (ABORT | HUNT))
        {
            rxdiscard (scc->l2port);
            scc->rx_len = 0;
            scc->rx_err = false;
        }

        if (delta & (Cpar[scc->l2port] ? HUNT : DCD))
        {
            if (scc->dcd = Cpar[scc->l2port] ? !(rr0 & HUNT) : rr0 & DCD)
            {
                if (!(scc->dcd_timer = Cpar[scc->l2port]))
                    set (rxinfo, scc->l2port);
            }
            else
            {
                res (rxinfo, scc->l2port);
                scc->dcd_timer = 0;
            }
        }
    }

    if (scc->clkchan && (scctype == OSCC || delta & CTS))
        ticks += cnt++ == 2 ? cnt = 0, 2 : 1;

    scc->rr0 = rr0;
    WR0 (RES_EX_INT);
}

/*** RX Handler *************************************************************/

void int_rx (scc_t *scc)
{
    byte rr8;

    rr8 = RR8;

    if (!scc->rx_err)
    {
        scc->rx_err = rxput (scc->l2port, rr8);
        scc->rx_len++;
    }
}

/*** Special RX Condition ***************************************************/

void int_sp (scc_t *scc)
{
    byte rr1;

    rr1 = RR (1);
    RR8;

    if (rr1 & RX_OVR)
    {
        scc->errors++;
        scc->rx_err = true;
    }

    if (rr1 & SDLC_EOF)
    {
        if (   !scc->rx_err
            && scc->rx_len >= MIN_LEN+1
            && (rr1 & (CRC_ERR | RESIDUE)) == RES8)
        {
            rxeof (scc->l2port, 1);
            l2call = true;
        }
        else
            if (scc->rx_len)
                rxdiscard (scc->l2port);

        scc->rx_len = 0;
        scc->rx_err = false;
    }

    WR0 (RES_ERR);
}

/*** SCC Hardware Reset *****************************************************/

void scc_hw_reset (void)
{
    scc_t *scc;
    word delay;

    for (scc = scctab; scc < &scctab[SCCCHANS]; scc += 2)
    {
        RR0;
        WR (9, HW_RES);
        for (delay = 100; delay--;);
    }
}

/*** SCC Init ***************************************************************/

void scc_init (void)
{
    scc_t *scc;

    if (clockdev == SCC)
        sccclkchan->clkchan = true;

    scc_hw_reset ();

    old_sccint = irq_init (sccirq, scc_int);

    for (scc = scctab; scc < sccend; scc++)
    {
        if (scc->modem)
        {
            if (scc->baud == 300)
                TApar[scc->l2port] = 4;

            WR (4, SDLC);
            WR (10, scc->modem == DF9IC ? NRZ | ABUNDER : NRZI | ABUNDER);
            WR (7, SDLC_FLAG);
            WR (3, RX_DIS);
            WR (5, TX_DIS);

            scc_clk (scc, false);

            WR (3, RX_EN);
            WR (5, scc->modem == SOFTCLK ? TX_DIS : TX_EN);

            WR (15, scc->clkchan ? SYNC_IE | DCD_IE | CTS_IE
                                 : SYNC_IE | DCD_IE);
            WR0 (RES_EX_INT);
            WR0 (RES_EX_INT);
            WR (1, EX_IE | TX_IE | RX_IE);
        }
        else
            if (scc->clkchan)
            {
                if (scctype == OSCC)
                {
                    WR (12, LOBYTE (TC75HZ));
                    WR (13, HIBYTE (TC75HZ));
                    WR (14, BRG_EN);

                    WR (15, ZC_IE);
                }
                else
                    WR (15, CTS_IE);

                WR0 (RES_EX_INT);
                WR0 (RES_EX_INT);
                WR (1, EX_IE);
            }

        scc->rr0 = RR0;
    }

    for (scc = scctab; scc < sccend; scc += 2)
    {
        WR (2, scc->chan / 2 * 0x10);
        WR (9, MINT_EN);
    }
}

/*** SCC Reset **************************************************************/

void scc_reset (void)
{
    irq_reset (sccirq, old_sccint);

    scc_hw_reset ();
}

/*** SCC Detect *************************************************************/

bool scc_detect (void)
{
    scc_t *scc;

    for (scc = scctab; scc < sccend; scc++)
    {
        RR0;
        if (   (WR (13, 0x55), RR (13) != 0x55)
            || (WR (13, 0xAA), RR (13) != 0xAA))
            return false;
    }

    return true;
}

/*** Modem RX ***************************************************************/

void modem_rx (modem_t *modem)
{
    if ((inp (modem->port_inp) ^ modem->level) & modem->data_inp)
    {
        modem->level ^= 0xFF;

        if (!Cpar[modem->l2port] || modem->dpll == 2)
        {
            if (modem->dcd_cnt < Cpar[modem->l2port])
                modem->dcd_cnt++;
            else
            {
                modem->dcd = true;
                set (rxinfo, modem->l2port);
            }
        }
        else
        {
            if (modem->dcd_cnt)
                modem->dcd_cnt--;
            else
            {
                modem->dcd = false;
                res (rxinfo, modem->l2port);
            }
        }

        modem->dpll = 0;

        if (modem->idle < STUFF)
        {
            modem->idle = 0;
            goto shift_bit;
        }

        if (modem->idle == FLAG)
        {
            if (modem->frm_len)
            {
                if (   modem->sync
                    && modem->frm_len >= MIN_LEN+2
                    && modem->bit_cnt == 1
                    && modem->fcs     == CRC_CHECK)
                {
                    rxeof (modem->l2port, 2);
                    l2call = true;
                }
                else
                    rxdiscard (modem->l2port);

                modem->frm_len = 0;
            }

            modem->bit_cnt = 8;
            modem->sync    = true;
            modem->crc     = CRC_RESET;
        }

        modem->idle = 0;
    }
    else
    {
        if (++modem->dpll == 4)
        {
            modem->dpll = 1;

            if (modem->idle < FREE)
            {
                if (++modem->idle < ABRT)
                {
shift_bit:          if (modem->sync)
                    {
                        if (modem->shift >>= 1, modem->dpll)
                            modem->shift |= 0x80;

                        modem->crc = (modem->crc >> 1) ^ ((modem->dpll ^
                                      modem->crc) & 1 ? CRC_MASK : 0);

                        if (--modem->bit_cnt == 0)
                        {
                            modem->sync    = !rxput (modem->l2port,
                                                     modem->shift);
                            modem->fcs     = modem->crc;
                            modem->bit_cnt = 8;
                            modem->frm_len++;
                        }
                    }
                }
                else
                    modem->sync = false;
            }
            else
            {
                modem->dcd = false;
                res (rxinfo, modem->l2port);
            }
        }
    }
}

/*** Modem TX ***************************************************************/

void modem_tx (modem_t *modem)
{
    if (modem->tx_state || modem->tx_kicked)
    {
        if (modem->tx_idle == STUFF && (modem->tx_state == ACTIVE
                                    ||  modem->tx_state == FLUSH))
        {
            outp (modem->port_out, inp (modem->port_out) ^ modem->data_out);
            modem->tx_idle = 0;
        }
        else
        {
            if (!modem->tx_bitcnt || modem->tx_state <= DWAIT)
            {
                switch (modem->tx_state)
                {
                case IDLE:
                    if (modem->dcd && !Dpar[modem->l2port])
                        return;

                    modem->tx_timer = Wpar[modem->l2port];
                    modem->tx_state = DWAIT;

                case DWAIT:
                    if (!Dpar[modem->l2port])
                    {
                        if (modem->tx_timer)
                            return;

                        if (modem->dcd || Ppar[modem->l2port] < random ())
                        {
                            modem->tx_state = IDLE;
                            return;
                        }
                    }

                    outp (modem->port_out, modem->ptt_on);
                    set (txinfo, modem->l2port);
                    modem->tx_timer = Tpar[modem->l2port];
                    modem->tx_state = DELAY;

                case DELAY:
                    if (modem->tx_timer)
                    {
                        modem->tx_shift  = SDLC_FLAG;
                        modem->tx_bitcnt = 8;
                        break;
                    }

                next_frm:
                    modem->tx_fcs   = CRC_RESET;
                    modem->tx_idle  = 0;
                    modem->tx_state = ACTIVE;

                case ACTIVE:
                    if ((modem->tx_shift = txget (modem->l2port)) < TXEOF)
                    {
                        modem->tx_bitcnt = 8;
                        break;
                    }

                    modem->tx_kicked = modem->tx_shift != TXLAST;
                    modem->tx_state  = FLUSH;
                    modem->tx_shift  = ~modem->tx_fcs;
                    modem->tx_bitcnt = 16;
                    break;

                case FLUSH:
                    modem->tx_shift  = SDLC_FLAG;
                    modem->tx_bitcnt = 8;
                    modem->tx_timer  = TApar[modem->l2port] + clkstep;
                    modem->tx_state  = TAIL;
                    break;

                case TAIL:
                    if (modem->tx_kicked)
                        goto next_frm;

                    if (modem->tx_timer)
                    {
                        modem->tx_shift  = SDLC_FLAG;
                        modem->tx_bitcnt = 8;
                        break;
                    }

                    outp (modem->port_out, modem->ptt_off);
                    res (txinfo, modem->l2port);
                    modem->tx_state = IDLE;
                    return;
                }
            }

            if (modem->tx_shift & 1)
                modem->tx_idle++;
            else
            {
                outp (modem->port_out,
                      inp (modem->port_out) ^ modem->data_out);
                modem->tx_idle = 0;
            }

            modem->tx_fcs = (modem->tx_fcs >> 1) ^ ((modem->tx_shift ^
                             modem->tx_fcs) & 1 ? CRC_MASK : 0);

            modem->tx_shift >>= 1;
            modem->tx_bitcnt--;
        }
    }
}

/*** Modem Init *************************************************************/

void modem_init (void)
{
    modem_t *modem;

    for (modem = modemtab; modem < modemend; modem++)
    {
        if (modem->baud == 300)
            TApar[modem->l2port] = 4;

        outp (modem->port_out, modem->ptt_off);
        if (modem->rs232)
            outp (modem->port_out - 1, 0x43);   /* BREAK (TxD +12V) */
    }

    outp (PIT, LOBYTE (pit0_const));            /* set time constant */
    outp (PIT, HIBYTE (pit0_const));
}

/*** Modem Reset ************************************************************/

void modem_reset (void)
{
    modem_t *modem;

    outp (PIT, 0);                              /* reset timer */
    outp (PIT, 0);

    for (modem = modemtab; modem < modemend; modem++)
    {
        outp (modem->port_out, 0);
        if (modem->rs232)
            outp (modem->port_out - 1, 3);      /* reset TxD */
    }
}

/*** RTC Interrupt **********************************************************/

void intr rtc_int (void)
{
    static byte step[] = {2, 1, 2, 1, 2, 1, 2, 2, 1, 2, 1, 2, 1, 2, 1, 2};
    static byte cnt;
    word newticks;

    eoi (8);

    if ((RDRTC (12) & 0xC0) == 0xC0)
    {
        ticks += newticks = step[cnt++ % 16];

        _enable ();
        timer (newticks);
    }
}

/*** RTC Init ***************************************************************/

void rtc_init (void)
{
    old_rtcint = irq_init (8, rtc_int);

    WRRTC (10, 0x2A);
    WRRTC (11, 0x42);
}

/*** RTC Reset **************************************************************/

void rtc_reset (void)
{
    WRRTC (11, 0x02);
    WRRTC (10, 0x26);

    irq_reset (8, old_rtcint);
}

/*** PAR RX *****************************************************************/

void par_rx (void)
{
    byte buf, bit, cnt = 16;

    while (cnt--)
    {
        bit = (buf = inp (partab.port_inp)) & RXD;
        outp (partab.port_out, PTT);
        outp (partab.port_out, PTT | BURST);

        partab.scr = (partab.scr << 1) | (bit != partab.lastbit);
        partab.lastbit = bit;

        bit =   !((word) partab.scr & 1)
              ^ !((word) partab.scr & 0x1000)
              ^ !(       partab.scr & 0x20000L);

        if (!bit)
        {
            if (partab.idle < STUFF)
            {
                partab.idle = 0;
                goto shift_bit;
            }

            if (partab.idle == FLAG)
            {
                if (   Cpar[partab.l2port]
                    && partab.dcd_cnt < DCDLIMIT
                    && ++partab.dcd_cnt == DCDCOUNT)
                {
                    partab.dcd = true;
                    partab.dcd_timer = Cpar[partab.l2port];
                }

                if (partab.frm_len)
                {
                    if (   partab.sync
                        && partab.frm_len >= MIN_LEN+2
                        && partab.bit_cnt == 1
                        && partab.fcs     == CRC_CHECK)
                    {
                        rxeof (partab.l2port, 2);
                        l2call = true;
                    }
                    else
                        rxdiscard (partab.l2port);

                    partab.frm_len = 0;
                }

                partab.bit_cnt = 8;
                partab.sync    = true;
                partab.crc     = CRC_RESET;
            }

            partab.idle = 0;
        }
        else
        {
            if (partab.idle < FREE)
            {
                if (++partab.idle < ABRT)
                {
shift_bit:          if (partab.sync)
                    {
                        if (partab.shift >>= 1, bit)
                            partab.shift |= 0x80;

                        partab.crc = (partab.crc >> 1) ^ ((bit ^
                                      partab.crc) & 1 ? CRC_MASK : 0);

                        if (--partab.bit_cnt == 0)
                        {
                            partab.sync    = !rxput (partab.l2port,
                                                     partab.shift);
                            partab.fcs     = partab.crc;
                            partab.bit_cnt = 8;
                            partab.frm_len++;
                        }
                    }
                }
                else
                {
                    if (Cpar[partab.l2port])
                    {
                        if (partab.dcd_cnt > DCDABORT)
                            partab.dcd_cnt -= DCDABORT;
                        else
                            partab.dcd_cnt = 0;

                        if (partab.dcd_cnt < DCDCOUNT)
                        {
                            partab.dcd = false;
                            partab.dcd_timer = 0;
                            res (rxinfo, partab.l2port);
                        }
                    }

                    partab.sync = false;
                }
            }
        }
    }

    if (!Cpar[partab.l2port])
        if (partab.dcd = buf & DCDBIT)
            set (rxinfo, partab.l2port);
        else
            res (rxinfo, partab.l2port);
}

/*** PAR TX *****************************************************************/

void par_tx (void)
{
    byte bit, cnt = 16;

    while (cnt--)
    {
        if (   partab.tx_idle == STUFF
            && (   partab.tx_state == ACTIVE
                || partab.tx_state == FLUSH))
        {
            partab.tx_bit ^= 1;
            partab.tx_idle = 0;
        }
        else
        {
            if (!partab.tx_bitcnt)
            {
                switch (partab.tx_state)
                {
                case DELAY:
                    if (partab.tx_timer)
                    {
                        partab.tx_shift  = SDLC_FLAG;
                        partab.tx_bitcnt = 8;
                        break;
                    }

                next_frm:
                    partab.tx_fcs   = CRC_RESET;
                    partab.tx_idle  = 0;
                    partab.tx_state = ACTIVE;

                case ACTIVE:
                    if ((partab.tx_shift = txget (partab.l2port)) < TXEOF)
                    {
                        partab.tx_bitcnt = 8;
                        break;
                    }

                    partab.tx_kicked = partab.tx_shift != TXLAST;
                    partab.tx_state  = FLUSH;
                    partab.tx_shift  = ~partab.tx_fcs;
                    partab.tx_bitcnt = 16;
                    break;

                case FLUSH:
                    partab.tx_shift  = SDLC_FLAG;
                    partab.tx_bitcnt = 8;
                    partab.tx_timer  = TApar[partab.l2port] + clkstep;
                    partab.tx_state  = TAIL;
                    break;

                case TAIL:
                    if (partab.tx_kicked)
                        goto next_frm;

                    if (partab.tx_timer)
                    {
                        partab.tx_shift  = SDLC_FLAG;
                        partab.tx_bitcnt = 8;
                        break;
                    }

                    outp (partab.port_out, PTT | BURST);
                    res (txinfo, partab.l2port);
                    partab.tx_state = IDLE;
                    return;
                }
            }

            if (partab.tx_shift & 1)
                partab.tx_idle++;
            else
            {
                partab.tx_bit ^= 1;
                partab.tx_idle = 0;
            }

            partab.tx_fcs = (partab.tx_fcs >> 1) ^ ((partab.tx_shift ^
                             partab.tx_fcs) & 1 ? CRC_MASK : 0);

            partab.tx_shift >>= 1;
            partab.tx_bitcnt--;
        }

        partab.scr =   (partab.scr << 1)
                     | (bit =    partab.tx_bit
                              ^ !((word) partab.scr & 0x800)
                              ^ !(       partab.scr & 0x10000L));

        outp (partab.port_out, bit);
        outp (partab.port_out, bit | BURST);
    }
}

/*** PAR Interrupt **********************************************************/

void intr par_int (void)
{
    debug (PAR);

    if (partab.tx_state > DWAIT)
        par_tx ();
    else
        par_rx ();

    eoi (partab.irq);
}

/*** PAR Init ***************************************************************/

void par_init (void)
{
    partab.old_int = irq_init (partab.irq, par_int);

    outp (partab.port_out, PTT | BURST);
    outp (partab.port_out + 2, inp (partab.port_out + 2) | IRQEN);
}

/*** PAR Reset **************************************************************/

void par_reset (void)
{
    outp (partab.port_out + 2, inp (partab.port_out + 2) & ~IRQEN);
    outp (partab.port_out, PTT | BURST);

    irq_reset (partab.irq, partab.old_int);
}

/*** CRC SMACK **************************************************************/

void crc_smack (word *crc, byte data)
{
    static word crctab[] =

   {0x0000, 0xC0C1, 0xC181, 0x0140, 0xC301, 0x03C0, 0x0280, 0xC241,
    0xC601, 0x06C0, 0x0780, 0xC741, 0x0500, 0xC5C1, 0xC481, 0x0440,
    0xCC01, 0x0CC0, 0x0D80, 0xCD41, 0x0F00, 0xCFC1, 0xCE81, 0x0E40,
    0x0A00, 0xCAC1, 0xCB81, 0x0B40, 0xC901, 0x09C0, 0x0880, 0xC841,
    0xD801, 0x18C0, 0x1980, 0xD941, 0x1B00, 0xDBC1, 0xDA81, 0x1A40,
    0x1E00, 0xDEC1, 0xDF81, 0x1F40, 0xDD01, 0x1DC0, 0x1C80, 0xDC41,
    0x1400, 0xD4C1, 0xD581, 0x1540, 0xD701, 0x17C0, 0x1680, 0xD641,
    0xD201, 0x12C0, 0x1380, 0xD341, 0x1100, 0xD1C1, 0xD081, 0x1040,
    0xF001, 0x30C0, 0x3180, 0xF141, 0x3300, 0xF3C1, 0xF281, 0x3240,
    0x3600, 0xF6C1, 0xF781, 0x3740, 0xF501, 0x35C0, 0x3480, 0xF441,
    0x3C00, 0xFCC1, 0xFD81, 0x3D40, 0xFF01, 0x3FC0, 0x3E80, 0xFE41,
    0xFA01, 0x3AC0, 0x3B80, 0xFB41, 0x3900, 0xF9C1, 0xF881, 0x3840,
    0x2800, 0xE8C1, 0xE981, 0x2940, 0xEB01, 0x2BC0, 0x2A80, 0xEA41,
    0xEE01, 0x2EC0, 0x2F80, 0xEF41, 0x2D00, 0xEDC1, 0xEC81, 0x2C40,
    0xE401, 0x24C0, 0x2580, 0xE541, 0x2700, 0xE7C1, 0xE681, 0x2640,
    0x2200, 0xE2C1, 0xE381, 0x2340, 0xE101, 0x21C0, 0x2080, 0xE041,
    0xA001, 0x60C0, 0x6180, 0xA141, 0x6300, 0xA3C1, 0xA281, 0x6240,
    0x6600, 0xA6C1, 0xA781, 0x6740, 0xA501, 0x65C0, 0x6480, 0xA441,
    0x6C00, 0xACC1, 0xAD81, 0x6D40, 0xAF01, 0x6FC0, 0x6E80, 0xAE41,
    0xAA01, 0x6AC0, 0x6B80, 0xAB41, 0x6900, 0xA9C1, 0xA881, 0x6840,
    0x7800, 0xB8C1, 0xB981, 0x7940, 0xBB01, 0x7BC0, 0x7A80, 0xBA41,
    0xBE01, 0x7EC0, 0x7F80, 0xBF41, 0x7D00, 0xBDC1, 0xBC81, 0x7C40,
    0xB401, 0x74C0, 0x7580, 0xB541, 0x7700, 0xB7C1, 0xB681, 0x7640,
    0x7200, 0xB2C1, 0xB381, 0x7340, 0xB101, 0x71C0, 0x7080, 0xB041,
    0x5000, 0x90C1, 0x9181, 0x5140, 0x9301, 0x53C0, 0x5280, 0x9241,
    0x9601, 0x56C0, 0x5780, 0x9741, 0x5500, 0x95C1, 0x9481, 0x5440,
    0x9C01, 0x5CC0, 0x5D80, 0x9D41, 0x5F00, 0x9FC1, 0x9E81, 0x5E40,
    0x5A00, 0x9AC1, 0x9B81, 0x5B40, 0x9901, 0x59C0, 0x5880, 0x9841,
    0x8801, 0x48C0, 0x4980, 0x8941, 0x4B00, 0x8BC1, 0x8A81, 0x4A40,
    0x4E00, 0x8EC1, 0x8F81, 0x4F40, 0x8D01, 0x4DC0, 0x4C80, 0x8C41,
    0x4400, 0x84C1, 0x8581, 0x4540, 0x8701, 0x47C0, 0x4680, 0x8641,
    0x8201, 0x42C0, 0x4380, 0x8341, 0x4100, 0x81C1, 0x8081, 0x4040};

    *crc = HIBYTE (*crc) ^ crctab[LOBYTE (*crc) ^ data];
}

/*** CRC RMNC ***************************************************************/

void crc_rmnc (word *crc, byte data)
{
    static word crctab[] =

   {0x0F87, 0x1E0E, 0x2C95, 0x3D1C, 0x49A3, 0x582A, 0x6AB1, 0x7B38,
    0x83CF, 0x9246, 0xA0DD, 0xB154, 0xC5EB, 0xD462, 0xE6F9, 0xF770,
    0x1F06, 0x0E8F, 0x3C14, 0x2D9D, 0x5922, 0x48AB, 0x7A30, 0x6BB9,
    0x934E, 0x82C7, 0xB05C, 0xA1D5, 0xD56A, 0xC4E3, 0xF678, 0xE7F1,
    0x2E85, 0x3F0C, 0x0D97, 0x1C1E, 0x68A1, 0x7928, 0x4BB3, 0x5A3A,
    0xA2CD, 0xB344, 0x81DF, 0x9056, 0xE4E9, 0xF560, 0xC7FB, 0xD672,
    0x3E04, 0x2F8D, 0x1D16, 0x0C9F, 0x7820, 0x69A9, 0x5B32, 0x4ABB,
    0xB24C, 0xA3C5, 0x915E, 0x80D7, 0xF468, 0xE5E1, 0xD77A, 0xC6F3,
    0x4D83, 0x5C0A, 0x6E91, 0x7F18, 0x0BA7, 0x1A2E, 0x28B5, 0x393C,
    0xC1CB, 0xD042, 0xE2D9, 0xF350, 0x87EF, 0x9666, 0xA4FD, 0xB574,
    0x5D02, 0x4C8B, 0x7E10, 0x6F99, 0x1B26, 0x0AAF, 0x3834, 0x29BD,
    0xD14A, 0xC0C3, 0xF258, 0xE3D1, 0x976E, 0x86E7, 0xB47C, 0xA5F5,
    0x6C81, 0x7D08, 0x4F93, 0x5E1A, 0x2AA5, 0x3B2C, 0x09B7, 0x183E,
    0xE0C9, 0xF140, 0xC3DB, 0xD252, 0xA6ED, 0xB764, 0x85FF, 0x9476,
    0x7C00, 0x6D89, 0x5F12, 0x4E9B, 0x3A24, 0x2BAD, 0x1936, 0x08BF,
    0xF048, 0xE1C1, 0xD35A, 0xC2D3, 0xB66C, 0xA7E5, 0x957E, 0x84F7,
    0x8B8F, 0x9A06, 0xA89D, 0xB914, 0xCDAB, 0xDC22, 0xEEB9, 0xFF30,
    0x07C7, 0x164E, 0x24D5, 0x355C, 0x41E3, 0x506A, 0x62F1, 0x7378,
    0x9B0E, 0x8A87, 0xB81C, 0xA995, 0xDD2A, 0xCCA3, 0xFE38, 0xEFB1,
    0x1746, 0x06CF, 0x3454, 0x25DD, 0x5162, 0x40EB, 0x7270, 0x63F9,
    0xAA8D, 0xBB04, 0x899F, 0x9816, 0xECA9, 0xFD20, 0xCFBB, 0xDE32,
    0x26C5, 0x374C, 0x05D7, 0x145E, 0x60E1, 0x7168, 0x43F3, 0x527A,
    0xBA0C, 0xAB85, 0x991E, 0x8897, 0xFC28, 0xEDA1, 0xDF3A, 0xCEB3,
    0x3644, 0x27CD, 0x1556, 0x04DF, 0x7060, 0x61E9, 0x5372, 0x42FB,
    0xC98B, 0xD802, 0xEA99, 0xFB10, 0x8FAF, 0x9E26, 0xACBD, 0xBD34,
    0x45C3, 0x544A, 0x66D1, 0x7758, 0x03E7, 0x126E, 0x20F5, 0x317C,
    0xD90A, 0xC883, 0xFA18, 0xEB91, 0x9F2E, 0x8EA7, 0xBC3C, 0xADB5,
    0x5542, 0x44CB, 0x7650, 0x67D9, 0x1366, 0x02EF, 0x3074, 0x21FD,
    0xE889, 0xF900, 0xCB9B, 0xDA12, 0xAEAD, 0xBF24, 0x8DBF, 0x9C36,
    0x64C1, 0x7548, 0x47D3, 0x565A, 0x22E5, 0x336C, 0x01F7, 0x107E,
    0xF808, 0xE981, 0xDB1A, 0xCA93, 0xBE2C, 0xAFA5, 0x9D3E, 0x8CB7,
    0x7440, 0x65C9, 0x5752, 0x46DB, 0x3264, 0x23ED, 0x1176, 0x00FF};

    *crc = (*crc << 8) ^ crctab[HIBYTE (*crc) ^ data];
}

/*** KISS Interrupt *********************************************************/

void kiss_int (kiss_t *kiss)
{
    byte iir, rx_buf;
    buf_t *parbuf;
    word tx_buf;

    debug (KISS);

    eoi (kiss->irq);

    while (!((iir = RDCOM (2)) & 1))
    {
        switch (iir & 6)
        {
        case 2:
lost_int:   if (kiss->fesc)
            {
                WRCOM (0, kiss->fesc);
                kiss->fesc = 0;
            }
            else
            {
                switch (kiss->tx_state)
                {
                case 1:
                    switch (kiss->tx_mode)
                    {
                    case CRCOFF:
                        WRCOM (0, DATA);
                        break;

                    case CRCSMACK:
                        kiss->tx_crc = 0xA001;
                        WRCOM (0, SMACK);
                        break;

                    case CRCRMNC:
                        kiss->tx_crc = 0xDEFD;
                        WRCOM (0, RMNC);
                    }
                    kiss->tx_state = 2;
                    break;

                case 2:
                    if ((tx_buf = txget (kiss->l2port)) < TXEOF)
                    {
                        switch (kiss->tx_mode)
                        {
                        case CRCSMACK:
                            crc_smack (&kiss->tx_crc, (byte) tx_buf);
                            break;

                        case CRCRMNC:
                            crc_rmnc (&kiss->tx_crc, (byte) tx_buf);
                        }

slip_out:               if (tx_buf == FEND || tx_buf == FESC)
                        {
                            kiss->fesc = tx_buf == FEND ? TFEND : TFESC;
                            tx_buf = FESC;
                        }

                        WRCOM (0, tx_buf);
                    }
                    else
                    {
                        kiss->tx_kicked = tx_buf != TXLAST;

                        if (kiss->tx_mode)
                        {
                            tx_buf =   kiss->tx_mode == CRCSMACK
                                     ? LOBYTE (kiss->tx_crc)
                                     : HIBYTE (kiss->tx_crc);
                            kiss->tx_state = 3;
                            goto slip_out;
                        }
                        else
                        {
                            WRCOM (0, FEND);
                            kiss->tx_state = 5;
                        }
                    }
                    break;

                case 3:
                    tx_buf =   kiss->tx_mode == CRCSMACK
                             ? HIBYTE (kiss->tx_crc)
                             : LOBYTE (kiss->tx_crc);
                    kiss->tx_state = 4;
                    goto slip_out;

                case 4:
                    WRCOM (0, FEND);
                    kiss->tx_state = 5;

                    if (!kiss->crc_ok)
                        switch (kiss->tx_mode)
                        {
                        case CRCSMACK:
                            kiss->tx_mode = CRCRMNC;
                            break;

                        case CRCRMNC:
                            kiss->tx_mode = CRCOFF;
                        }
                    break;

                case 5:
                    if (!kiss->parlst)
                    {
kiss_cont:              if (kiss->tx_kicked)
                        {
                            WRCOM (0, FEND);
                            kiss->tx_state = 1;
                        }
                        else
                        {
                            res (txinfo, kiss->l2port);
                            kiss->tx_state = 0;
                        }
                        break;
                    }
                    kiss->tx_state = 6;

                case 6:
                    if (kiss->bufp == 4)
                    {
                        kiss->bufp = 0;

                        parbuf = kiss->parlst;
                        kiss->parlst = parbuf->next;
                        dealoc (parbuf);

                        if (!kiss->parlst)
                            goto kiss_cont;
                    }

                    WRCOM (0, kiss->parlst->data[kiss->bufp++]);
                }
            }
            break;

        case 4:
            rx_buf = RDCOM (0);

            set (rxinfo, kiss->l2port);
            kiss->dcd_timer = not0 (Cpar[kiss->l2port]);

            if (!(RDCOM (5) & 0x1E))
            {
                switch (kiss->rx_state)
                {
                case 0:
                    if (rx_buf == FEND)
                        kiss->rx_state = 1;
                    break;

                case 1:
                    switch (rx_buf)
                    {
                    case FEND:
                        break;

                    case DATA:
                    case SMACK:
                    case RMNC:
                        if (   !kiss->crc_ok
                            ||    kiss->tx_mode == CRCSMACK
                               && rx_buf        == SMACK
                            ||    kiss->tx_mode == CRCRMNC
                               && rx_buf        == RMNC)
                        {
                            switch (rx_buf)
                            {
                            case DATA:
                                kiss->rx_mode = CRCOFF;
                                break;

                            case SMACK:
                                kiss->rx_mode = CRCSMACK;
                                kiss->rx_crc  = 0xA001;
                                break;

                            case RMNC:
                                kiss->rx_mode = CRCRMNC;
                                kiss->rx_crc  = 0xDEFD;
                            }

                            kiss->rx_len   = 0;
                            kiss->rx_state = 2;
                            break;
                        }

                    default:
                        kiss->rx_state = 0;
                    }
                    break;

                case 2:
                    switch (rx_buf)
                    {
                    case FEND:
                        if (kiss->rx_mode)
                        {
                            if (   kiss->rx_len < MIN_LEN+2
                                ||    kiss->rx_mode == CRCSMACK
                                   && kiss->rx_crc  != 0
                                ||    kiss->rx_mode == CRCRMNC
                                   && kiss->rx_crc  != 0x7070)
                                goto kiss_err;

                            kiss->crc_ok  = true;
                            kiss->tx_mode = kiss->rx_mode;

                            rxeof (kiss->l2port, 2);
                        }
                        else
                        {
                            if (kiss->rx_len < MIN_LEN)
                                goto kiss_err;

                            rxeof (kiss->l2port, 0);
                        }

                        l2call = true;
                        kiss->rx_state = kiss->crc_ok ? 1 : 0;
                        break;

                    case FESC:
                        kiss->rx_state = 3;
                        break;

                    default:
kiss_put:               kiss->rx_len++;

                        switch (kiss->rx_mode)
                        {
                        case CRCSMACK:
                            crc_smack (&kiss->rx_crc, rx_buf);
                            break;

                        case CRCRMNC:
                            crc_rmnc (&kiss->rx_crc, rx_buf);
                        }

                        if (rxput (kiss->l2port, rx_buf))
                            goto kiss_err;
                    }
                    break;

                case 3:
                    if (rx_buf == TFEND || rx_buf == TFESC)
                    {
                        rx_buf = rx_buf == TFEND ? FEND : FESC;
                        kiss->rx_state = 2;
                        goto kiss_put;
                    }
                    else
                        goto kiss_err;
                }
            }
            else
            {
kiss_err:       rxdiscard (kiss->l2port);
                kiss->rx_state = 0;
                kiss->errors++;
            }
        }
    }

    if (kiss->tx_state && (RDCOM (5) & 0x20))
        goto lost_int;
}

void intr kiss_int0 (void) { kiss_int (&kisstab[0]); }

void intr kiss_int1 (void) { kiss_int (&kisstab[1]); }

void intr kiss_int2 (void) { kiss_int (&kisstab[2]); }

void intr kiss_int3 (void) { kiss_int (&kisstab[3]); }

/*** KISS Init **************************************************************/

procfp hook[] = {kiss_int0, kiss_int1, kiss_int2, kiss_int3};

void kiss_init (void)
{
    kiss_t *kiss;
    word tconst;

    for (kiss = kisstab; kiss < kissend; kiss++)
    {
        kiss->tx_mode = CRCSMACK;

        WRCOM (1, 0);
        WRCOM (4, 0);

        kiss->old_int = irq_init (kiss->irq, hook[kiss - kisstab]);

        tconst = 115200L / kiss->baud;

        WRCOM (3, 0x83);
        WRCOM (0, LOBYTE (tconst));
        WRCOM (1, HIBYTE (tconst));
        WRCOM (3, 0x03);

        while (RDCOM (5) & 1)
            RDCOM (0);

        WRCOM (4, 0x0B);
        WRCOM (1, 0x03);

        l1par (kiss->l2port, TPAR,  defTp);
        l1par (kiss->l2port, PPAR,  defPp);
        l1par (kiss->l2port, WPAR,  defWp);
        l1par (kiss->l2port, DPAR,  defDp);
    }
}

/*** KISS Reset *************************************************************/

void kiss_reset (void)
{
    kiss_t *kiss;

    for (kiss = kisstab; kiss < kissend; kiss++)
    {
        WRCOM (1, 0);
        WRCOM (4, 0);

        irq_reset (kiss->irq, kiss->old_int);
    }
}

/*** Interrupt Handlers *****************************************************/

void intr int8 (void)
{
    static word tick_cnt = 1,
                int8_cnt = 1;
    modem_t *modem;

    debug (MODEM);

    for (modem = modemtab; modem < modemend; modem++)
    {
        if (--modem->baud_cnt == 0)
        {
            modem->baud_cnt = modem->baud_delay;

            if (--modem->send_cnt == 0)
            {
                modem->send_cnt = 3;
                modem_tx (modem);               /* 1200/s */
            }

            if (modem->tx_state <= DWAIT || Dpar[modem->l2port])
                modem_rx (modem);               /* 3600/s */
        }
    }

    _enable ();
    eoi (0);

    if (clockdev == MODEM && --tick_cnt == 0)
    {
        tick_cnt = tick_delay;
        ticks += 2;                             /* 50/s */

        timer (2);
    }

    if (--int8_cnt == 0)
    {
        int8_cnt = int8_delay;
        chain (old_int8);                       /* 18.2/s */
    }
}

void intr int8scc (void)
{
    chain (old_int8);
}

void intr int13 (void)
{
    chain (old_int13);
}

void intr int1C (void)
{
    static byte cnt;

    debug (LOOP);

    if (clockdev == LOOP)
    {
        ticks += cnt = cnt == 5 ? 6 : 5;

        _enable ();
        timer (cnt);
    }

    chain (old_int1C);
}

void intr int21 (void)
{
     chain (old_int21);
}

/*** Init Interrupts ********************************************************/

void init_intr (void)
{
    _disable ();

    old_int8   = _dos_getvect (0x08);
    old_int13  = _dos_getvect (0x13);
    old_int1C  = _dos_getvect (0x1C);
    old_int21  = _dos_getvect (0x21);
    old_defInt = _dos_getvect (defInt);

    _dos_setvect (0x08, modems ? int8 : int8scc);
    _dos_setvect (0x13, int13);
    _dos_setvect (0x1C, int1C);
    _dos_setvect (0x21, int21);
    _dos_setvect (defInt, drsi ? iface_drsi : iface_tfpc);

    if (modems)    modem_init ();
    if (parports)  par_init ();
    if (sccs)      scc_init ();
    if (kissports) kiss_init ();

    if (clockdev == RTC)
        rtc_init ();

    if (l1ports && !modems)
        irq_prio (2);

    _enable ();
}

/*** Reset Interrupts *******************************************************/

void _cdecl reset_intr (void)
{
    _disable ();

    if (l1ports && !modems)
        irq_prio (0);

    if (clockdev == RTC)
        rtc_reset ();

    if (kissports) kiss_reset ();
    if (sccs)      scc_reset ();
    if (parports)  par_reset ();
    if (modems)    modem_reset ();

    _dos_setvect (0x08,   old_int8);
    _dos_setvect (0x13,   old_int13);
    _dos_setvect (0x1C,   old_int1C);
    _dos_setvect (0x21,   old_int21);
    _dos_setvect (defInt, old_defInt);

    _enable ();
}

/*** Memory Sizes ***********************************************************/

byte *minmem (void)
{
    return bufbgn;
}

byte *maxmem (void)
{
    return bufend - 1;
}

/*** Host <-> TF ************************************************************/

bool ishget (void)
{
    return hihead != hitail;
}

byte hgetc (void)
{
    return hibuf[hitail++ & HOSTBUF-1];
}

bool ishput (void)
{
    return hohead != hotail;
}

void hputc (byte ch)
{
    if (hohead - hotail < HOSTBUF)
        hobuf[hohead++ & HOSTBUF-1] = ch;
}

/*** Reset ******************************************************************/

void reset (void)
{
    *(wordfp) 0x400072 = 0x1234;            /* warm boot */
    ((void (_far *) ()) 0xFFFF0000) ();
}

/*** Keep it resident *******************************************************/

void keep (word retcode)
{
    bytefp endp = (bytefp) bufend;

    _dos_freemem (*(wordfp) (((dword) _psp << 16) + 0x2C));
    _dos_keep (retcode,
              (word) ((dword) FP_OFF (endp)+15 >> 4) + FP_SEG (endp) - _psp);
}

/*** TFPCX loaded? **********************************************************/

byte hdr_tfpc[] = "???N5NX",
     hdr_drsi[] = "\x53\x1E\xBB??\x8E\xDB\x84\xE4\x74\x20";

bool loaded (void)
{
    byte vector, hdr, *p1;
    bytefp p2, handler;

    for (vector = 0xFF; vector >= 0x40; vector--)
    {
        handler = (bytefp) _dos_getvect (vector);

        for (hdr = 0, p1 = hdr_tfpc; hdr < 2; hdr++, p1 = hdr_drsi)
            for (p2 = handler; *p1 == '?' || *p1 == *p2; p2++)
                if (!*++p1)
                {
                    defInt = vector;
                    drsi   = (bool) hdr;
                    return true;
                }
    }

    return false;
}

/*** Error Message **********************************************************/

enum {ERR_OPT,    ERR_PORT,   ERR_MANY, ERR_DETECT, ERR_ADDR, ERR_INT,
      ERR_NOLOAD, ERR_LOADED, ERR_VER,  ERR_BAUD,   ERR_FILE, ERR_NUM,
      ERR_IRQ,    ERR_MODE,   ERR_BUF,  ERR_CHAN,   ERR_MTASK};

char *msg[] = {"Invalid option (-H for help)",
               "Invalid port",
               "Too many ports",
               "Port not present",
               "Port address out of range",
               "Interrupt out of range",
               "TFPCX not loaded",
               "TFPCX already loaded",
               "Invalid version number",
               "Invalid baud rate",
               "Can't open file",
               "Invalid numerical format",
               "IRQ out of range",
               "Invalid clock mode",
               "Number of buffers out of range",
               "Number of channels out of range",
               "Serial modems don't work in this system"};

void errmsg (byte err)
{
    cprintf ("\nError: %s\r\n", msg[err]);
    exit (EXIT_FAILURE);
}

/*** Signon *****************************************************************/

void signon (void)
{
    byte port;

    cprintf ("Ŀ\r\n"
             " TFPCX v"__VER__" ("__DATE__") by DG0FT \r\n"
             "      TheFirmware by NORD><LINK      \r\n"
             "Ĵ\r\n"
             " This is free software without any   \r\n"
             " warranty. You can redistribute it   \r\n"
             " under the terms of the GNU General  \r\n"
             " Public License (see LICENSE.DOC).   \r\n"
             "Ĵ\r\n"
             " %u Port(s), %2u Channels, %s-Int %X \r\n",
             l1ports, linknmbr, drsi ? "DRSI" : "TFPC", defInt);

    if (l1ports)
    {
        cputs ("Ĵ\r\n");

        for (port = 0; port < l1ports; port++)
            cprintf (" %u: %s (%X/%02u), %5u Bd, %-7s \r\n",
                     port, l1info (port, L1NAME), l1info (port, L1ADDR),
                           l1info (port, L1IRQ),  l1info (port, L1BAUD),
                           l1info (port, L1TYPE));
    }

    cputs ("\r\n");
}

/*** Token ******************************************************************/

char *token (char *str)
{
    static char *pos;

    if (str)
        pos = str;
    else
        str = pos;

    while (*pos && *pos != ':')
        pos++;

    if (*pos)
        *pos++ = '\0';

    return str;
}

/*** Number *****************************************************************/

word number (char *str, byte base)
{
    dword value = 0;
    byte digit;

    if (*(str = token (str)))
    {
        while (*str)
        {
            if ((digit = *str++ - '0') > 9)
                error (ERR_NUM, (digit -= 7) < 10 || digit >= base);

            error (ERR_NUM, (value = value * base + digit) > 0xFFFF);
        }

        error (ERR_NUM, !value);
    }

    return (word) value;
}

/*** Get Address ************************************************************/

word getaddr (word defaddr)
{
    word addr;

    if (addr = number (NULL, 16))
    {
        error (ERR_ADDR,    addr < 0x100
                         || addr > 0x3F8
                         || addr & 7);
    }
    else
        error (ERR_DETECT, !(addr = defaddr));

    return addr;
}

/*** Get IRQ ****************************************************************/

byte getirq (byte defirq)
{
    byte irq;

    if (irq = number (NULL, 10))
    {
        error (ERR_IRQ,    irq  < 2
                        || irq == 6
                        || irq == 8
                        || irq == 13
                        || irq  > 15);
        if (isat ())
        {
            if (irq == 2) irq = 9;
        }
        else
            error (ERR_IRQ, irq > 7);
    }
    else
        irq = defirq;

    return irq;
}

/*** Add Port ***************************************************************/

byte addport (byte major, void *minor)
{
    error (ERR_MANY, l1ports == L1PORTS);

    l1port[l1ports].major = lastdev = major;
    l1port[l1ports].minor = minor;

    return l1ports++;
}

/*** Set Timer **************************************************************/

void settimer (byte major, byte prio, byte step)
{
    if (prio > clkprio)
    {
        clockdev = major;
        clkprio  = prio;
        clkstep  = step;
    }
}

/*** Modem Config ***********************************************************/

void modem_cfg (char *arg)
{
    modem_t *modem;

    error (ERR_MANY, modems++ == MODEMS);

    modem->l2port = addport (MODEM, modem = modemend++);

    error (ERR_PORT,    (modem->rs232 = strncmp (arg, "LPT", 3))
                     &&                 strncmp (arg, "COM", 3)
                     || (modem->port = number (arg+3, 10) - 1) > 3
                     || arg[4]);

    modem->port_out = getaddr (BIOSBASE[modem->port]);

    if (modem->rs232)
    {
        modem->port_inp = 2 + (modem->port_out += 4);
        modem->ptt_on   = 2;
        modem->ptt_off  = 1;
        modem->data_out = 1;
        modem->data_inp = 0x10;
    }
    else
    {
        modem->port_inp = 1 + modem->port_out;
        modem->ptt_on   = 0xBF;
        modem->ptt_off  = 0x3F;
        modem->data_out = 0x40;
        modem->data_inp = 0x80;
    }

    settimer (MODEM, 100, 2);
}

/*** PAR Config *************************************************************/

void par_cfg (char *arg)
{
    error (ERR_MANY, parports++);

    partab.l2port = addport (PAR, &partab);

    error (ERR_PORT,    strncmp (arg, "PAR", 3)
                     || (partab.port = number (arg+3, 10) - 1) > 3
                     || arg[4]);

    partab.port_out = getaddr (LPTBASE[partab.port]);
    partab.port_inp = partab.port_out + 1;

    partab.irq = getirq (7);
}

/*** SCC Config *************************************************************/

sccboard_t sccboard[] = {{0x300, 7, "2222", {1, 0, 3, 2}, 16},  /* DSCC */
                         {0x150, 3, "4445", {3, 1, 7, 5}, -1},  /* OSCC */
                         {0x300, 7, "1103", {0, 1, 2, 3}, 4}};  /* USCC */

void scc_cfg (char *arg)
{
    byte chan, lastchan, modem;
    sccboard_t *board;
    word sccbase;
    char *modems;
    scc_t *scc;

    error (ERR_MANY, sccs);
    error (ERR_PORT, strcmp (token (arg+1), "SCC"));

    board = &sccboard[scctype];

    sccbase = getaddr (board->base);

    sccirq = getirq (board->irq);

    if (!*(modems = token (NULL)))
        modems = board->modems;

    for (chan = 0, scc = scctab; chan < SCCCHANS; chan++, scc++)
    {
        scc->chan = chan;
        scc->data = sccbase + board->data[chan];
        scc->ctrl = scc->data + board->ctrl;

        if (*modems)
        {
            error (ERR_MODE, (modem = *modems++ - '0') > CLKCHAN);

            if (modem)
            {
                if (modem == CLKCHAN)
                {
                    error (ERR_MODE, scctype != OSCC);
                    settimer (SCC, 75, 2);
                    sccclkchan = scc;
                }
                else
                {
                    scc->l2port = addport (SCC, scc);
                    scc->modem = modem;
                    sccs++;
                }

                lastchan = chan;
            }
        }
    }

    error (ERR_MODE, *modems || !sccs);

    sccend = &scctab[(lastchan + 2) & 0xFE];

    switch (scctype)
    {
    case USCC: sccclkchan = &scctab[1];
    case DSCC: settimer (SCC, 75, 2);
    }
}

/*** KISS Config ************************************************************/

void kiss_cfg (char *arg)
{
    kiss_t *kiss;

    error (ERR_MANY, kissports++ == KISSPORTS);

    kiss->l2port = addport (KISS, kiss = kissend++);

    error (ERR_PORT,    strncmp (arg, "KISS", 4)
                     || (kiss->port = number (arg+4, 10) - 1) > 3
                     || arg[5]);

    kiss->base = getaddr (COMBASE[kiss->port]);

    kiss->irq = getirq ((byte) (4 - (kiss->port & 1)));
}

/*** Baud Config ************************************************************/

void baud_cfg (char *arg)
{
    word baud, maxbaud = 0;
    modem_t *modem;
    port_t *port;
    scc_t *scc;
    byte prt;

    for (prt = 0, port = l1port; prt < l1ports; prt++, port++, arg = NULL)
    {
        baud = number (arg, 10);

        switch (port->major)
        {
        case MODEM:
            if (!baud)
                baud = 1200;

            error (ERR_BAUD,    baud !=  300 && baud != 1200
                             && baud != 2400 && baud != 4800);

            if ((port->modem->baud = baud) > maxbaud)
                maxbaud = baud;

            break;

        case PAR:
            if (!baud)
                baud = 9600;

            error (ERR_BAUD, baud != 9600 && baud != 19200);

            partab.baud = baud;

            break;

        case SCC:
            scc = port->scc;

            if (!baud)
                baud =    scc->modem == SOFTCLK
                       || scc->modem == PA0HZP ? 1200 : 9600;

            error (ERR_BAUD,    scc->modem != DF9IC
                             && (baud < 50 || baud > 38400));

            scc->baud = baud;

            break;

        case KISS:
            if (!baud)
                baud = 9600;

            error (ERR_BAUD,    baud !=  2400 && baud !=  4800
                             && baud !=  9600 && baud != 19200
                             && baud != 38400 && baud != 57600);

            port->kiss->baud = baud;
        }
    }

    for (modem = modemtab; modem < modemend; modem++)
    {
        modem->baud_delay =
        modem->baud_cnt   = maxbaud / modem->baud;
        modem->send_cnt   = 3;
    }

    switch (maxbaud)
    {
    case  300: pit0_const = 1325; int8_delay =  49; tick_delay =  18; break;
    case 1200: pit0_const =  331; int8_delay = 198; tick_delay =  72; break;
    case 2400: pit0_const =  166; int8_delay = 395; tick_delay = 144; break;
    case 4800: pit0_const =   83; int8_delay = 790; tick_delay = 288; break;
    }
}

/*** Help *******************************************************************/

char help[] =

"\n"
"Usage: TFPCX [ -N ] [ <load options> | -T | -U ]\r\n"
"\n"
"<general options>              <legend>\r\n"
"  -N no messages                 [] optional\r\n"
"  -T terminal mode               |  alternative\r\n"
"  -U unload                      x  hex digit\r\n"
"                                 n  dec digit\r\n"
"<load options>\r\n"
"  -P<port>[:xxx:nn:nnnn]  packet port [addr:IRQ:<clock>]\r\n"
"  -Bnnnn[:nnnn ...]       baud rate (1 number/port)\r\n"
"\n"
"  -F[file]  read init file      -D  debug mode\r\n"
"  -C[xx]    show DCD [color]    -DM use DRSI messages\r\n"
"  -Ixx      TFPCX interrupt     -DR emulate DRSI driver\r\n"
"  -BU[nnnn] number of buffers   -DX modified DRSI interface\r\n"
"  -CHnn     number of channels  -NB no blinking rectangle\r\n"
"  -ST       time stamp          -NL no loopback\r\n"
"\n"
"<port>  COMn | LPTn | PARn | KISSn | DSCC | OSCC | USCC (n = 1-4)\r\n"
"\n"
"<clock> 0 = disable    2 = hardclock    4 = PA0HZP port  (1 digit/\r\n"
"        1 = softclock  3 = DF9IC modem  5 = PA0HZP timer  channel)\r\n";

/*** Read Init **************************************************************/

void read_init (word handle)
{
    bool empty = true, ignore = false;
    word spaces = 0;
    byte ch;

    if (logo)
        cputs ("\nReading ...\r\n\n");

    while (read (handle, &ch, 1) == 1)
    {
        if (ch == '\n')
        {
            if (!empty)
                tfpcx_out ('\r');

            empty  = true;
            ignore = false;
            spaces = 0;
        }
        else
            if (!ignore)
                switch (ch)
                {
                case ' ':
                case '\t':
                    spaces++;
                    break;

                case '#':
                case ';':
                    ignore = true;
                    break;

                case '^':
                    if (empty)
                        break;

                default:
                    if (empty)
                    {
                        tfpcx_out (defESC);

                        empty  = false;
                        spaces = 0;
                    }
                    else
                        for (; spaces; spaces--)
                            tfpcx_out (' ');

                    tfpcx_out (ch);
                }

        while (tfpcx_avl ())
        {
            ch = tfpcx_inp ();

            if (logo)
                putch (ch);
        }
    }

    close (handle);
}

/*** Set Envp ***************************************************************/

void _cdecl _setenvp (void)         /* suppress copying environment */
{
}

/*** Main *******************************************************************/

void main (word argc, char **argv)
{
    char ch, *baudarg = "", *initfile = NULL;
    word bufsize, handle;

    while (*++argv)
    {
        error (ERR_OPT, *strupr((*argv)++) != '-');

        switch (*(*argv)++)
        {
        case 'B':
            if (**argv == 'U')
                bufcnt = number (*argv+1, 10);
            else
                baudarg = *argv;
            break;

        case 'C':
            if (**argv == 'H')
                linknmbr = defYp = number (*argv+1, 10);
            else
            {
                attrib  = (byte) number (*argv, 16);
                carrier = true;
            }
            break;

        case 'D':
            switch (**argv)
            {
            case 'R':   drmsg    = true;
            case 'X':   drsi     = true;
            case 'M':   dxmsg    = true;    break;
            case '\0':  debugdev = lastdev; break;

            default:    errmsg (ERR_OPT);
            }
            break;

        case 'F':
            initfile = **argv ? *argv : "TFPCX.INI";
            break;

        case 'H':
            cputs (help);
            exit (EXIT_SUCCESS);

        case 'I':
            defInt = number (*argv, 16);
            error (ERR_INT, defInt < 0x40 || 0xFF < defInt);
            break;

        case 'N':
            switch (**argv)
            {
            case 'B':   blink    = false; break;
            case 'L':   loopback = false; break;
            case '\0':  logo     = false; break;

            default:    errmsg (ERR_OPT);
            }
            break;

        case 'P':
            switch (**argv)
            {
            case 'U': scctype++;
            case 'O': scctype++;
            case 'D': scc_cfg   (*argv); break;
            case 'K': kiss_cfg  (*argv); break;
            case 'P': par_cfg   (*argv); break;
            default:  modem_cfg (*argv);
            }
            break;

        case 'S':
            error (ERR_OPT, **argv != 'T');
            stampok = true;
            break;

        case 'T':
            error (ERR_NOLOAD, !loaded ());

            if (logo)
                cputs ("\nTerminal mode, Press ALT-X to quit!\r\n\n");

            for (;;)
            {
                while (tfpcx_avl ())
                    putch (tfpcx_inp ());

                if (kbhit ())
                    if (ch = getch ())
                        tfpcx_out (ch);
                    else
                        if (getch () == 0x2D)      /* ALT-X */
                            exit (EXIT_SUCCESS);
            }

        case 'U':
            error (ERR_NOLOAD, !loaded ());
            error (ERR_VER, call_tfpcx (0xFE00) != VERSION);
            _dos_freemem (call_tfpcx (0xFF00));

            if (logo)
                cputs ("\nTFPCX unloaded\r\n");
            exit (EXIT_SUCCESS);

        default:
            errmsg (ERR_OPT);
        }
    }

    if (isat () && !mtasker ())
        settimer (RTC, 64, 2);

    error (ERR_LOADED, loaded ());

    error (ERR_CHAN,    linknmbr < 4
                     || linknmbr > LINKNMBR
                     || !(lnktbl = malloc (linknmbr * LB_SIZE)));

    if (!bufcnt)
        bufcnt = _memmax () / MB_SIZE;
    error (ERR_BUF,    bufcnt < 400
                    || bufcnt > 0xFFFF / MB_SIZE
                    || !(bufbgn = malloc (bufsize = bufcnt * MB_SIZE)));
    bufend = bufbgn + bufsize;

    baud_cfg (baudarg);

    error (ERR_MTASK, modems && mtasker ());

    error (ERR_DETECT, sccs && !scc_detect ());

    if (initfile)
        error (ERR_FILE, (handle = open (initfile, O_RDONLY | O_TEXT)) == -1);

    if (logo)
        signon ();

    mainf ();
    init_intr ();

    if (initfile)
        read_init (handle);

    keep (EXIT_SUCCESS);
}

/****************************************************************************/
