/************************************************************************/
/*                                                                      */
/*    *****                       *****                                 */
/*      *****                   *****                                   */
/*        *****               *****                                     */
/*          *****           *****                                       */
/*  ***************       ***************                               */
/*  *****************   *****************                               */
/*  ***************       ***************                               */
/*          *****           *****           TheNetNode                  */
/*        *****               *****         Portable                    */
/*      *****                   *****       Network                     */
/*    *****                       *****     Software                    */
/*                                                                      */
/* File src/graph.c (maintained by: DG9AML)                             */
/*                                                                      */
/* This file is part of "TheNetNode" - Software Package                 */
/*                                                                      */
/* Copyright (C) 1998  NORD><LINK e.V. Braunschweig                     */
/*                                                                      */
/* This program is free software; you can redistribute it and/or modify */
/* it under the terms of the NORD><LINK ALAS (Allgemeine Lizenz fuer    */
/* Amateurfunk Software) as published by Hans Georg Giese (DF2AU)       */
/* on 13/Oct/1992; either version 1, or (at your option) any later      */
/* version.                                                             */
/*                                                                      */
/* This program is distributed WITHOUT ANY WARRANTY only for further    */
/* development and learning purposes. See the ALAS (Allgemeine Lizenz   */
/* fuer Amateurfunk Software).                                          */
/*                                                                      */
/* You should have received a copy of the NORD><LINK ALAS (Allgemeine   */
/* Lizenz fuer Amateurfunk Software) along with this program; if not,   */
/* write to NORD><LINK e.V., Hinter dem Berge 5, D-38108 Braunschweig   */
/*                                                                      */
/* Dieses Programm ist PUBLIC DOMAIN, mit den Einschraenkungen durch    */
/* die ALAS (Allgemeine Lizenz fuer Amateurfunk Software), entweder     */
/* Version 1, veroeffentlicht von Hans Georg Giese (DF2AU),             */
/* am 13.Oct.1992, oder (wenn gewuenscht) jede spaetere Version.        */
/*                                                                      */
/* Dieses Programm wird unter Haftungsausschluss vertrieben, aus-       */
/* schliesslich fuer Weiterentwicklungs- und Lehrzwecke. Naeheres       */
/* koennen Sie der ALAS (Allgemeine Lizenz fuer Amateurfunk Software)   */
/* entnehmen.                                                           */
/*                                                                      */
/* Sollte dieser Software keine ALAS (Allgemeine Lizenz fuer Amateur-   */
/* funk Software) beigelegen haben, wenden Sie sich bitte an            */
/* NORD><LINK e.V., Hinter dem Berge 5, D-38108 Braunschweig            */
/*                                                                      */
/************************************************************************/

#include "tnn.h"

#ifdef GRAPH
static void showgraph(int, int, int, int, MBHEAD *);
static void graph_actualpeak(TGRAPHPEAK *, ULONG);
static void graph_actual(TGRAPHELEMENT *);
static void graph_average(TGRAPHELEMENT *, ULONG);
static void graph_clear_element(TGRAPHELEMENT *);

static void graph_clear_element(TGRAPHELEMENT *element)
{
  TGRAPHPEAK dummy;
  dummy.max =
  dummy.ave = 0L;
  dummy.min = 0xffffffffL;

  element->freebuffer = dummy;
  element->circuit = dummy;
  element->l2link = dummy;
  element->node = dummy;
  element->roundpsec = dummy;
  element->throughput = dummy;
}

static void graph_actualpeak(TGRAPHPEAK *select, ULONG newvalue)
{
  if (select->max < newvalue) select->max = newvalue;
  if (select->min > newvalue) select->min = newvalue;
  select->ave += newvalue;
}

static void graph_actual(TGRAPHELEMENT *select)
{
  graph_actualpeak(&select->freebuffer, nmbfre);
  graph_actualpeak(&select->circuit   , nmbcir);
  graph_actualpeak(&select->l2link    , nmblks);
  graph_actualpeak(&select->node      , netp->num_nodes);
  graph_actualpeak(&select->roundpsec , rounds_pro_sec);
  graph_actualpeak(&select->throughput, thbps * 8L);
}

static void graph_average(TGRAPHELEMENT *select, ULONG factor)
{
  if (factor > 0) {
    select->freebuffer.ave /= factor;
    select->circuit   .ave /= factor;
    select->l2link    .ave /= factor;
    select->node      .ave /= factor;
    select->roundpsec .ave /= factor;
    select->throughput.ave /= factor;
  }
}

/**************************************************************************/
/* GRAPH                                                                  */
/*                                                                        */
/* Funktion: zeigt einen Textgrahpen an. Zur Graphengestaltung werden     */
/*           nur niederwertige Asciis verwendet.                          */
/**************************************************************************/
#define MAXULONG 0xffffffffL
#define GRAPH_ALLSELECT  0
#define GRAPH_ERRSELECT  1
#define GRAPH_CIRCUIT    2
#define GRAPH_FREEBUFFER 3
#define GRAPH_L2LINK     4
#define GRAPH_NODE       5
#define GRAPH_ROUNDPSEC  6
#define GRAPH_THROUGHPUT 7

#define GRAPH_HOUR       1
#define GRAPH_DAY        2
#define GRAPH_WEEK       3

TGRAPH graph;

/*------------------------------------------------------------------------*/
void ccpgraph(void)
{
  MBHEAD   *mbp;
  int      timesel, select, expand = 0, presentation = 0;
  char     chr;

  mbp = getmbp();
  putchr('\r',mbp);

  /* nur der Sysop darf Daten loeschen */
  if (issyso() && strnicmp((char *)clipoi, "CLEAR", 5) == 0) {
    putstr("Graph cleared!\r", mbp);
    prompt(mbp);
    seteom(mbp);
    graphclear();
    return;
  }
  chr = toupper(*clipoi);
  switch (chr) {
    case 'H': timesel = GRAPH_HOUR;break;
    case 'D': timesel = GRAPH_DAY;break;
    case 'W': timesel = GRAPH_WEEK;break;
    default : timesel = 0;
  }

  if (timesel) {
    clipoi++;
    clicnt--;
    skipsp(&clicnt, &clipoi);
    chr = toupper(*clipoi);
  } else timesel = GRAPH_HOUR;

  switch (chr) {
    case 'B': select = GRAPH_THROUGHPUT; break;
    case 'C': select = GRAPH_CIRCUIT; break;
    case 'F': select = GRAPH_FREEBUFFER; break;
    case 'L': select = GRAPH_L2LINK; break;
    case 'N': select = GRAPH_NODE; break;
    case 'R': select = GRAPH_ROUNDPSEC; break;
    case '*': select = GRAPH_ALLSELECT; break;
    default : select = GRAPH_ERRSELECT; break;
  }

  /* Presentationmode feststellen */
  clipoi++;
  clicnt--;
  skipsp(&clicnt, &clipoi);
  chr = toupper(*clipoi);
  switch (chr) {
    case '1': presentation = 1; break;
    case '2': presentation = 2; break;
    default : presentation = 0;
  }
    /* Expandmode feststellen */
  if (presentation != 0) {
    clipoi++;
    clicnt--;
    skipsp(&clicnt, &clipoi);
    chr = toupper(*clipoi);
  }
  if (clicnt > 0) {
    if (chr == '+') expand = 1;
    else select = GRAPH_ERRSELECT;
  }

  if (select == GRAPH_ERRSELECT) {
    putstr("(G)raph (H)our (B)aud\r"
           "        (D)ay  (C)ircuits\r"
           "        (W)eek (F)ree buffers\r"
           "               (L)2-Links\r"
           "               (N)odes\r"
           "               (R)ounds\r"
           "               (*) All\r",mbp);
   prompt(mbp);
   seteom(mbp);
   return;
  }
  if (select == GRAPH_ALLSELECT) {
    showgraph(timesel, GRAPH_THROUGHPUT, presentation, expand, mbp);
    showgraph(timesel, GRAPH_CIRCUIT, presentation, expand, mbp);
    showgraph(timesel, GRAPH_FREEBUFFER, presentation, expand, mbp);
    showgraph(timesel, GRAPH_L2LINK, presentation, expand, mbp);
    showgraph(timesel, GRAPH_NODE, presentation, expand, mbp);
    showgraph(timesel, GRAPH_ROUNDPSEC, presentation, expand, mbp);
  }
  else showgraph(timesel, select, presentation, expand, mbp);
  prompt(mbp);
  seteom(mbp);
}

/**************************************************************************/
/* Funktion showgraph()                                                   */
/*                                                                        */
/* int timesel:      GRAPH_HOUR: Ueberblick 1 Stunde                      */
/*                   GRAPH_DAY:  Ueberblick 24 Stunden                    */
/*                   GRAPH_WEEK: Ueberblick 7 Tage                        */
/* int select:       Buffer,BPS,etc.  ist angewaehlt                      */
/* int presentation: 0: Min-/Mittel-/Maxwert-Darstellung                  */
/*                   1: Alternative Dars. (ohne Min/Max Unterscheidung)   */
/*                   2: Mittelwert-Darstellung                            */
/* int expand:  0: Diagramm nicht dehnen                                  */
/*              1: Diagramm zwischen Max und Min strecken                 */
/* MBHEAD *mhp: Stringuebergabe (Output)                                  */
/**************************************************************************/
static void showgraph(int timesel, int select, int presentation,
                      int expand,MBHEAD *mbp)
{
  TGRAPHELEMENT element[100];
  TGRAPHPEAK       peak[100];
  ULONG raster = 0L;                    /* noetig fuer Y-Achse (Werteachse) */
  ULONG wertmax,wertave,wertmin;
  ULONG maximum,minimum,average;
  ULONG expandwert;                            /* enthaelt minimum oder 0 */
  WORD  lastline = 1;            /* bei expand = 1 , die 0. Zeile mitausgaben */
  WORD  elements,slot,pos;
  WORD  averagenum;
  int   lines = GRAPH_LINES;
  int   i,j;
  char  zeile[128];                        /* Leerzeilenballast entfernen */

  switch (timesel) {
    case  GRAPH_DAY:
            elements = GRAPH_DAY_ELEMENTS;
            for (i = 0; i < elements; i++) element[i] = graph.day[i];
            slot = graph.day_slot;
            pos = graph.day_pos;
            break;
    case  GRAPH_WEEK:
            elements = GRAPH_WEK_ELEMENTS;
            for (i = 0; i < elements; i++) element[i] = graph.week[i];
            slot = graph.week_slot;
            pos = graph.week_pos;
            break;
    default: elements = GRAPH_STD_ELEMENTS;
            for (i = 0; i < elements; i++) element[i] = graph.hour[i];
            slot = graph.hour_slot;
            pos = graph.hour_pos;
  }

  switch (select) {
    case GRAPH_FREEBUFFER:
           putstr("Free buffers:", mbp);
           for (i = 0; i < elements; i++) peak[i] = element[i].freebuffer;
           break;
    case GRAPH_CIRCUIT:
           putstr("Circuits:",mbp);
           for (i = 0; i < elements; i++) peak[i] = element[i].circuit;
           break;
    case GRAPH_L2LINK:
           putstr("L2-Links:", mbp);
           for (i = 0; i < elements; i++) peak[i] = element[i].l2link;
           break;
    case GRAPH_NODE:
           putstr("Nodes:", mbp);
           for (i = 0; i < elements; i++) peak[i] = element[i].node;
           break;
    case GRAPH_ROUNDPSEC:
           putstr("Rounds per seconds:", mbp);
           for (i = 0; i < elements; i++) peak[i] = element[i].roundpsec;
           break;
    case GRAPH_THROUGHPUT:
           putstr("Throughput (Baud):", mbp);
           for (i = 0; i < elements; i++) peak[i] = element[i].throughput;
           break;
    default:
           select = GRAPH_FREEBUFFER;
           putstr("Free buffers:", mbp);
           for (i = 0; i < elements; i++) peak[i] = element[i].freebuffer;
  }
  /* Maximum,Mittel und Minimum Wert bestimmen */
  maximum = average = 0L;
  minimum = MAXULONG;
  averagenum = 0;

  for (i = 0; i < elements; i++)
  { wertmax = peak[i].max;
    wertave = (i != pos) ? peak[i].ave : (slot > 0) ? peak[i].ave / slot : 0;
    wertmin = peak[i].min;

    if (maximum < wertmax && wertmax != MAXULONG) maximum = wertmax;
    if (minimum > wertmin) minimum = wertmin;
    if (wertave != MAXULONG) {
       average += wertave;
       averagenum++;
    }
  } /* for elements */

  if (minimum == MAXULONG) minimum = 0;
  if (maximum == MAXULONG) maximum = 0;
  if (maximum > 0)  putprintf(mbp, "  Maximum: %lu", (ULONG)maximum);
  if (averagenum > 0 && average > 0)
                  putprintf(mbp, "  Average: %lu",(ULONG)average / averagenum);
                  putprintf(mbp, "  Minimum: %lu\r\r",(ULONG)minimum);

  /* expand = 1: Alle Zeilen unter Minimum weggelassen. Es stehen somit     */
  /* 5 bis GRAPH_LINES Zeilen fuer die Werte zwischen Min und Max         */
  /* zur Verfuegung.                                                      */
  if (expand == 1) maximum -= minimum;

  /* Zeilenraster ermitteln */
  raster = maximum / GRAPH_LINES;
  if (raster == 0 || maximum % GRAPH_LINES > 0) raster++;

  /* genaue Zeilenanzahl ermitteln */
  lines = maximum / raster;            /* schwankt zwischen 5 und GRAPH_LINES */
  if (maximum % raster > 0) lines++;
  if (lines < 5) lines = 5;                   /* min. sollen es 5 Zeilen sein */

  /* Damit Minimalzeile im Expandmode mitangezeigt wird */
  if (expand == 1 && minimum > 0) lastline = 0; else lastline = 1;
  lines++;
  if (expand == 1) expandwert = minimum; else expandwert = 0;
  /* Die einzelnen Zeilen generieren */
  do
  { lines--;
    zeile[0] = '\0';
    putprintf(mbp, "%6lu|", (ULONG) raster * lines + expandwert);
    for (i = 0; i < elements; i++)
    { if (timesel == GRAPH_HOUR)
        j = (i + pos + 1 >= GRAPH_STD_ELEMENTS) ?
                         pos + i + 1 - GRAPH_STD_ELEMENTS : i + pos + 1;
      else
        j = i;

      wertmax = peak[j].max;
      wertave = (j != pos) ? peak[j].ave : (slot > 0) ? peak[j].ave / slot : 0;
      wertmin = peak[j].min;

      switch (presentation) {
               /* Alternative Darstellung (ohne Min / Max Unterscheidung)*/
        case 1: if (   wertave >= raster * lines + expandwert
                    && wertave != MAXULONG)
                {  putstr(zeile, mbp);
                   putchr('#', mbp);
                   zeile[0] = '\0';
                }
                else strcat(zeile, " ");
                break;
                /* Mittelwert-Bereich */
        case 2: if (   wertave >= raster * lines + expandwert
                    && wertave != MAXULONG
                    && wertmin < raster * lines + expandwert
                    && wertmin != MAXULONG)
                {  putstr(zeile, mbp);
                   putchr('*', mbp);
                   zeile[0] = '\0';
                }
                else strcat(zeile, " ");
                break;
                /* Min-/Mittel-/Max-Bereich */
       default: if (   wertmax >= raster * lines + expandwert
                    && wertmax != MAXULONG) {
                  putstr(zeile, mbp);
                  if (   wertave >= raster * lines + expandwert
                      && wertave != MAXULONG) {
                    if (   wertmin >= raster * lines + expandwert
                        && wertmin != MAXULONG)
                      putchr('#', mbp);
                    else
                      putchr('*', mbp);
                  }
                  else putchr('+', mbp);
                  zeile[0] = '\0';
                }
                else strcat(zeile, " ");
                break;
      } /* presentation-switch */
    } /* for */

    putchr('\r', mbp);

  } while (lines != lastline);  /* naechste Zeile */

  /* untere Zeile generieren */
  putprintf(mbp, "%7s", "+");

  for (i = 0; i < elements; i++)
     if ((timesel == GRAPH_DAY  && i == graph.day_pos) ||
         (timesel == GRAPH_WEEK && i == graph.week_pos)) putchr('A', mbp);
     else putchr('-', mbp);
  switch (timesel) {
    case GRAPH_DAY:
              putprintf(mbp, "\r%7s00  02  04  06  08  10  "
                             "12  14  16  18  20  22\r"
                             "%9s01  03  05  07  09  11  "
                             "13  15  17  19  21  23 Hour\r\r", " " ," ");
              break;
    case GRAPH_WEEK:
              putprintf(mbp, "\r%7sMonday  Tuesday Wednes. Thursd. "
                             "Friday  Saturd. Sunday\r\r", " ");
              break;
   default:   putprintf(mbp, " Elap. time\r%6s-3600 -3240 -2880 "
                             "-2520 -2160 -1800 -1440 -1080 -720  "
                             "-360  0 Seconds\r\r"," ");
  } /* switch */
}

/**************************************************************************/
/* Funktion graphclear(void)                                              */
/*                                                                        */
/* loescht die Graphdaten                                                 */
/**************************************************************************/
void graphclear(void)
{
  int i;
  TGRAPHELEMENT dummy2;
  TGRAPHPEAK dummy;
  graph.hour_pos = 0;
  graph.day_pos = 0;
  graph.week_pos = 0;
  graph.hour_slot = 0;
  graph.day_slot = 0;
  graph.week_slot = 0;

  dummy.max =
  dummy.ave =
  dummy.min = MAXULONG;

  dummy2.circuit    = dummy;
  dummy2.freebuffer = dummy;
  dummy2.l2link     = dummy;
  dummy2.node       = dummy;
  dummy2.roundpsec  = dummy;
  dummy2.throughput = dummy;

  for (i = 0; i < GRAPH_STD_ELEMENTS; i++) graph.hour[i] = dummy2;
  for (i = 0; i < GRAPH_DAY_ELEMENTS; i++) graph.day [i] = dummy2;
  for (i = 0; i < GRAPH_WEK_ELEMENTS; i++) graph.week[i] = dummy2;

  graph.enabled = FALSE;     /* wird erst nach einer Minute auf TRUE gesetzt */
}

void graph_timer(void)
{
  static LONG    graph_time = 0L;
  int            min, hour, day;

  graph_time--;

  /* Alle GRAPH_INTERVAL Sekunden werden die Daten verarbeitet */
  if (graph_time <= 0L && graph.enabled == TRUE)
  {
     graph_time = GRAPH_INTERVAL;          /* Wartezeit neu setzen */

     min  = sys_localtime->tm_min;
     hour = sys_localtime->tm_hour;
     day  = (sys_localtime->tm_wday != 0) ? sys_localtime->tm_wday - 1 : 6;

     /* Jede Minute ein neues Datenfeld vorbeiten */
     /* Feldposition aktualisieren         dg9aml */

     if (graph.hour_pos != min) {
        /* altes Feld ist abgeschlossen, kann gemittelt werden */
        graph_average(&graph.hour[graph.hour_pos], graph.hour_slot);
        /* neues Feld vorbereiten, Werte zuruecksetzen */
        graph.hour_slot = 0;
        graph.hour_pos = min;
        graph_clear_element(&graph.hour[graph.hour_pos]);
     }
     /* Jede halbe Stunde ein neues Datenfeld vorbeiten dg9aml*/
     if (graph.day_pos != 2 * hour + min / 30)
     {
        graph_average(&graph.day[graph.day_pos], graph.day_slot);
        graph.day_slot = 0;
        graph.day_pos = 2 * hour + min / 30;
        graph_clear_element(&graph.day[graph.day_pos]);
     }

     /* Alle 3 Stunden ein neues Datenfeld vorbeiten */
     if (graph.week_pos != 8 * day + hour / 3)
     {
        graph_average(&graph.week[graph.week_pos], graph.week_slot);
        graph.week_slot = 0;
        graph.week_pos = 8 * day + hour / 3;
        graph_clear_element(&graph.week[graph.week_pos]);
     }
     graph_actual(&graph.hour[graph.hour_pos]);
     graph.hour_slot++;
     graph_actual(&graph.day[graph.day_pos]);
     graph.day_slot++;
     graph_actual(&graph.week[graph.week_pos]);
     graph.week_slot++;
  } /* if graph.enabled */
}
#endif

/* End of src/graph.c */
